
###############################################################################################################
###############################################################################################################

#' Function to calculate an estimate of the score for each contribution to the likelihood

#' @param datU  n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
#' @param par parameters of the model (delta, log(lambda), nu)
#' @param thresh quantile at which fit was censored, for example 0.95
#' @param coord D*2 matrix of spatial coordinates
#' @param eps value of epsilon for numerical differencing (differences are taken at range 2*epsilon)
#' @param QMC logical. If TRUE then faster implementation of QMC multivariate normal is used.
#' @param ... additional parameters to be passed on when QMC=TRUE. Specifically "prime" and "ncore" can be changed from the defaults of 127 and 1, respectively
#'
#' @return n x 3 matrix containing the estimated score contribution for each parameter (columns) and datapoint (rows). Ordering is: delta, log(lambda), nu
#' @details This calculates an estimate of the score function for each contribution to the likelihood. The main use is likely to be in the case of temporal dependence, in which case the user will likely want to aggregate score contributions over a certain time period (e.g. a month, year etc.) and use these contributions to estimate the variance of the score. This can then be combined with an estimated Hessian to get an estimate of the sandwich covariance matrix
#' @author Jenny Wadsworth
#'
#' @export


score.est<-function(datU,par,thresh,coord,eps,QMC=TRUE,...)
{
  if(QMC)
  {
    eval1.p<-fit.cop.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0,0),optim=FALSE,sum=FALSE,...)
    eval1.m<-fit.cop.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par-c(eps,0,0),optim=FALSE,sum=FALSE,...)

    eval2.p<-fit.cop.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps,0),optim=FALSE,sum=FALSE,...)
    eval2.m<-fit.cop.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,eps,0),optim=FALSE,sum=FALSE,...)

    eval3.p<-fit.cop.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,0,eps),optim=FALSE,sum=FALSE,...)
    eval3.m<-fit.cop.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,0,eps),optim=FALSE,sum=FALSE,...)
  } else{
  eval1.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0,0),optim=FALSE,sum=FALSE)
  eval1.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(eps,0,0),optim=FALSE,sum=FALSE)

  eval2.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps,0),optim=FALSE,sum=FALSE)
  eval2.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,eps,0),optim=FALSE,sum=FALSE)

  eval3.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,0,eps),optim=FALSE,sum=FALSE)
  eval3.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,0,eps),optim=FALSE,sum=FALSE)
  }

  score.est<-cbind(eval1.p-eval1.m,eval2.p-eval2.m,eval3.p-eval3.m)/(2*eps)

  return(score.est)
}

#
# variance.score<-function(datU,par,thresh,coord,eps)
# {
#   eval1.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0,0),optim=FALSE,sum=FALSE)
#   eval1.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(eps,0,0),optim=FALSE,sum=FALSE)
#
#   eval2.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps,0),optim=FALSE,sum=FALSE)
#   eval2.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,eps,0),optim=FALSE,sum=FALSE)
#
#   eval3.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,0,eps),optim=FALSE,sum=FALSE)
#   eval3.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,0,eps),optim=FALSE,sum=FALSE)
#
#   score.est<-cbind(eval1.p-eval1.m,eval2.p-eval2.m,eval3.p-eval3.m)/(2*eps)
#   var.score<-var(score.est)*dim(datU)[1]
#
#   return(list(score.est=score.est,var.score=var.score))
# }


###############################################################################################################
###############################################################################################################

# #' Function to calculate the sandwich covariance matrix, in case of temporal dependence
#
# #' @param datU  n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
# #' @param par parameters of the model (delta, log(lambda), nu)
# #' @param thresh quantile at which fit was censored, for example 0.95
# #' @param coord D*2 matrix of spatial coordinates
# #' @param eps value of epsilon for numerical differencing (differences are taken at range 2*epsilon)
# #' @param suppliedHessian Hessian matrix already obtained through other methods. If not supplied, then an estimate is calculated by the function.
# #'
# #' @return list with objects score.est (estimate of the score), var.score (variance of score matrix), hessian (estimated / supplied Hessian), and sandwich (estimated sandwich covariance matrix)
# #' @author Jenny Wadsworth
# #'
# #' @export
#
# sandwich.cov<-function(datU,par,thresh,coord,eps,suppliedHessian=NULL)
# {
#   eval1.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0,0),optim=FALSE,sum=FALSE)
#   eval1.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(eps,0,0),optim=FALSE,sum=FALSE)
#
#   eval2.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps,0),optim=FALSE,sum=FALSE)
#   eval2.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,eps,0),optim=FALSE,sum=FALSE)
#
#   eval3.p<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,0,eps),optim=FALSE,sum=FALSE)
#   eval3.m<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,0,eps),optim=FALSE,sum=FALSE)
#
#   score.est<-cbind(eval1.p-eval1.m,eval2.p-eval2.m,eval3.p-eval3.m)/(2*eps)
#
#   if(is.matrix(suppliedHessian))
#   {
#     var.score<-var(score.est)*dim(datU)[1]
#     sandwich<-try(solve(suppliedHessian)%*%var.score%*%solve(suppliedHessian))
#     return(list(score.est=score.est,var.score=var.score,hessian=suppliedHessian,sandwich=sandwich))
#   }
#   else{
#     eval0<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par,optim=FALSE)
#
#     # summations needed in eval1.p--eval3.m as these return vectors
#     H11<-(sum(eval1.p)+sum(eval1.m)-2*eval0)/eps^2
#     H22<-(sum(eval2.p)+sum(eval2.m)-2*eval0)/eps^2
#     H33<-(sum(eval3.p)+sum(eval3.m)-2*eval0)/eps^2
#
#     eval12.pp<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,eps,0),optim=FALSE)
#     eval12.pm<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,-eps,0),optim=FALSE)
#     eval12.mp<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(-eps,eps,0),optim=FALSE)
#     eval12.mm<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(-eps,-eps,0),optim=FALSE)
#
#     H12<-(eval12.pp+eval12.mm-eval12.pm-eval12.mp)/(4*eps^2)
#
#     eval13.pp<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0,eps),optim=FALSE)
#     eval13.pm<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0,-eps),optim=FALSE)
#     eval13.mp<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(-eps,0,eps),optim=FALSE)
#     eval13.mm<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(-eps,0,-eps),optim=FALSE)
#
#     H13<-(eval13.pp+eval13.mm-eval13.pm-eval13.mp)/(4*eps^2)
#
#     eval23.pp<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps,eps),optim=FALSE)
#     eval23.pm<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps,-eps),optim=FALSE)
#     eval23.mp<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,-eps,eps),optim=FALSE)
#     eval23.mm<-fit.cop.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,-eps,-eps),optim=FALSE)
#
#     H23<-(eval23.pp+eval23.mm-eval23.pm-eval23.mp)/(4*eps^2)
#
#     H<-matrix(0,ncol=3,nrow=3)
#     diag(H)<-c(H11,H22,H33)
#     H[1,2]<-H[2,1]<-H12
#     H[1,3]<-H[3,1]<-H13
#     H[2,3]<-H[3,2]<-H23
#
#     var.score<-var(score.est)*dim(datU)[1]
#     sandwich<-try(solve(H)%*%var.score%*%solve(H))
#
#     return(list(score.est=score.est,var.score=var.score,hessian=H,sandwich=sandwich))
#   }
# }


###############################################################################################################
###############################################################################################################

#' Function to calculate an estimate of the score for each contribution to the likelihood for the Gaussian copula

#' @param datU  n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
#' @param par parameters of the Gaussian model (log(lambda), nu)
#' @param thresh quantile at which fit was censored, for example 0.95
#' @param coord D*2 matrix of spatial coordinates
#' @param eps value of epsilon for numerical differencing (differences are taken at range 2*epsilon)
#' @param QMC logical. If TRUE, faster QMC computation of the multivarite Gaussian probabilities is used.
#' @param ... additional arguments ("prime") to be passed on if QMC=TRUE
#'
#' @return n x 2 matrix containing the estimated score contribution for each parameter (columns) and datapoint (rows). Ordering is: log(lambda), nu
#' @author Jenny Wadsworth
#'
#' @export

score.est.Gauss<-function(datU,par,thresh,coord,eps,QMC=TRUE,...)
{
  if(QMC)
  {
    eval1.p<-fit.Gauss.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0),optim=FALSE,sum=FALSE,...)
    eval1.m<-fit.Gauss.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par-c(eps,0),optim=FALSE,sum=FALSE,...)

    eval2.p<-fit.Gauss.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps),optim=FALSE,sum=FALSE,...)
    eval2.m<-fit.Gauss.Ddim.QMC(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,eps),optim=FALSE,sum=FALSE,...)
  } else{
  eval1.p<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0),optim=FALSE,sum=FALSE)
  eval1.m<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(eps,0),optim=FALSE,sum=FALSE)

  eval2.p<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps),optim=FALSE,sum=FALSE)
  eval2.m<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,eps),optim=FALSE,sum=FALSE)
  }
  score.est<-cbind(eval1.p-eval1.m,eval2.p-eval2.m)/(2*eps)
  return(score.est)
}

# #' Function to calculate the sandwich covariance matrix for the Gaussian model, in case of temporal dependence
#
# #' @param datU  n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
# #' @param par parameters of the Gaussian model (log(lambda), nu)
# #' @param thresh quantile at which fit was censored, for example 0.95
# #' @param coord D*2 matrix of spatial coordinates
# #' @param eps value of epsilon for numerical differencing (differences are taken at range 2*epsilon)
# #' @param suppliedHessian Hessian matrix already obtained through other methods. If not supplied, then an estimate is calculated by the function.
# #'
# #' @return list with objects score.est (estimate of the score), var.score (variance of score matrix), hessian (estimated / supplied Hessian), and sandwich (estimated sandwich covariance matrix)
# #' @author Jenny Wadsworth
# #'
# #' @export
#
# sandwich.cov.Gauss<-function(datU,par,thresh,coord,eps,suppliedHessian=NULL)
# {
#   eval1.p<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,0),optim=FALSE,sum=FALSE)
#   eval1.m<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(eps,0),optim=FALSE,sum=FALSE)
#
#   eval2.p<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(0,eps),optim=FALSE,sum=FALSE)
#   eval2.m<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par-c(0,eps),optim=FALSE,sum=FALSE)
#
#   score.est<-cbind(eval1.p-eval1.m,eval2.p-eval2.m)/(2*eps)
#
#   if(is.matrix(suppliedHessian))
#   {
#     var.score<-var(score.est)*dim(datU)[1]
#     sandwich<-try(solve(suppliedHessian)%*%var.score%*%solve(suppliedHessian))
#     return(list(score.est=score.est,var.score=var.score,hessian=suppliedHessian,sandwich=sandwich))
#   }
#   else{
#     eval0<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par,optim=FALSE)
#
#     # summations needed in eval1.p--eval3.m as these return vectors
#     H11<-(sum(eval1.p)+sum(eval1.m)-2*eval0)/eps^2
#     H22<-(sum(eval2.p)+sum(eval2.m)-2*eval0)/eps^2
#
#     eval12.pp<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,eps),optim=FALSE)
#     eval12.pm<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(eps,-eps),optim=FALSE)
#     eval12.mp<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(-eps,eps),optim=FALSE)
#     eval12.mm<-fit.Gauss.Ddim(datU=datU,coord = coord,thres = thresh,init.val=par+c(-eps,-eps),optim=FALSE)
#
#     H12<-(eval12.pp+eval12.mm-eval12.pm-eval12.mp)/(4*eps^2)
#
#     H<-matrix(0,ncol=2,nrow=2)
#     diag(H)<-c(H11,H22)
#     H[1,2]<-H[2,1]<-H12
#
#     var.score<-var(score.est)*dim(datU)[1]
#     sandwich<-try(solve(H)%*%var.score%*%solve(H))
#
#     return(list(score.est=score.est,var.score=var.score,hessian=H,sandwich=sandwich))
#   }
# }


###############################################################################################################
###############################################################################################################


# joint survival function ***on diagonal only*** (will return an incorrect result if unequal values given for x
# as inclusion-exclusion formulae will come into play)

pGbar.diag<-function(x,delta,sigma,rel.tol){
  if(!is.matrix(x)){
    x <- matrix(x,nrow=1)
  }
  D <- ncol(x)
  pGi <- function(xi){
    fun <- function(p,delta){
      Y1 <- xi/(1-delta)+delta*log(1-p)/(1-delta)
      Y2 <- Y1
      Y1small <- Y1<qexp(0.5)
      Y2[Y1small] <- qnorm(1-exp(-Y1[Y1small]))
      Y2[!Y1small] <- -qnorm(exp(-Y1[!Y1small]))
      set.seed(123)
      return(pmvnorm(lower=Y2,sigma=sigma)[1])
    }
    fun<-Vectorize(fun,vectorize.args = "p")
    if(is.null(rel.tol)){
    if(delta <= 0.55){
      rel.tol <- 10^(-4)
    } else if(delta > 0.55 & delta <= 0.75){
      rel.tol <- 10^(-5)
    } else if(delta > 0.75 & delta <= 0.85){
      rel.tol <- 10^(-7)
    } else if(delta > 0.85){
      rel.tol <- 10^(-8)
    }}
    res <- integrate(fun,lower=0,upper=pexp(min(xi)/delta),delta,rel.tol=rel.tol,stop.on.error=FALSE)$value
    return(res)
  }
  logres <- log(apply(x,1,pGi))
  # on the diagonal only, the d-dim joint survivor function has a simple form:
  return( exp(logres) + exp(-x[1]/delta) )

}


###############################################################################################################
###############################################################################################################


pCbar.diag<-function(u,delta,sigma,rel.tol){
    return( pGbar.diag(qG1(u,delta),delta,sigma,rel.tol=rel.tol) )
  }



###############################################################################################################
###############################################################################################################

#' calculates model-based chi_u in D dimensions, assuming W to have a Gaussian copula with powered exponential covariance function
#' @param  u sequence of values at which to calculate chi
#' @param cov.par scale and shape parameter (respectively) of the powered exponential covariance function
#' @param dist distance matrix between sites
#' @param delta value of delta
#'
#' @return returns a vector corresponding to the value of chiu at each value of u
#' @author Raphael Huser and Jenny Wadsworth
#'
#' @export

chiuD <- function(u,cov.par,dist,delta,rel.tol=NULL){
  chis <- rep(NA,length(u))
  sigma <- exp(-(dist/cov.par[1])^cov.par[2])
  for(i in 1:length(u)){
    cat(paste(rep("\b",15),collapse=""))
    cat(paste(c(round(i/length(u),2))*100, "%"))
      try( chis[i] <- (pCbar.diag(rep(u[i],dim(dist)[1]),delta,sigma,rel.tol=rel.tol))/(1-u[i]) )
  }
  return(chis)
}



###############################################################################################################
###############################################################################################################
# chiuD.QMC - QMC version of chiuD

#' calculates model-based chi_u in D dimensions, assuming W to have a Gaussian copula with powered exponential covariance function. Uses
#' faster QMC computation than chiuD.

#' @param  u sequence of values at which to calculate chi
#' @param cov.par scale and shape parameter (respectively) of the powered exponential covariance function
#' @param dist distance matrix between sites
#' @param delta value of delta
#' @param prime prime number for the QMC calculation of multivariate Gaussian probabilities
#'
#' @details Behaves like chiuD, except typically much faster
#' @references de Fondeville and Davison (2016) High-dimensional peaks-over-threshold inference for the Brown-Resnick process. https://arxiv.org/abs/1605.08558
#' @return returns a vector corresponding to the value of chiu at each value of u
#' @author Raphael Huser and Jenny Wadsworth
#'
#' @export
#'
chiuD.QMC<-function(u,cov.par,dist,delta,prime=127,rel.tol=NULL){
  require(mvPot)
  chis <- rep(NA,length(u))
  sigma <- exp(-(dist/cov.par[1])^cov.par[2])
  for(i in 1:length(u)){
    cat(paste(rep("\b",15),collapse=""))
    cat(paste(c(round(i/length(u),2))*100, "%"))
    try( chis[i] <- (pCbar.diag.QMC(rep(u[i],dim(dist)[1]),delta,sigma,prime=prime,rel.tol = rel.tol))/(1-u[i]) )
  }
  return(chis)
}

pCbar.diag.QMC<-function(u,delta,sigma,prime,rel.tol){
  return( pGbar.diag.QMC(qG1(u,delta),delta,sigma,prime=prime,rel.tol=rel.tol) )
}

pGbar.diag.QMC<-function(x,delta,sigma,prime,rel.tol){
  require(mvPot)
  if(!is.matrix(x)){
    x <- matrix(x,nrow=1)
  }
  D <- ncol(x)
  pGi <- function(xi){
    fun <- function(p,delta){
      Y1 <- xi/(1-delta)+delta*log(1-p)/(1-delta)
      Y2 <- Y1
      Y1small <- Y1<qexp(0.5)
      Y2[Y1small] <- qnorm(1-exp(-Y1[Y1small]))
      Y2[!Y1small] <- -qnorm(exp(-Y1[!Y1small]))
      set.seed(123)
      vec <- genVecQMC(prime, D)
      #set.seed(seed = seed)
      return(mvPot::mvtNormQuasiMonteCarlo(upper=-Y2,cov=sigma,p=vec$primeP,genVec=vec$genVec)[1])
      # return(pmvnorm(lower=Y2,sigma=sigma)[1])
    }
    fun<-Vectorize(fun,vectorize.args = "p")
    if(is.null(rel.tol)){
     if(delta <= 0.55){
       rel.tol <- 10^(-4)
     } else if(delta > 0.55 & delta <= 0.75){
       rel.tol <- 10^(-5)
     } else if(delta > 0.75 & delta <= 0.85){
       rel.tol <- 10^(-7)
     } else if(delta > 0.85){
       rel.tol <- 10^(-8)
     }}
    res <- integrate(fun,lower=0,upper=pexp(min(xi)/delta),delta,rel.tol=rel.tol,stop.on.error=FALSE)$value
    return(res)
  }
  logres <- log(apply(x,1,pGi))
  # on the diagonal only, the d-dim joint survivor function has a simple form:
  return( exp(logres) + exp(-x[1]/delta) )

}

###############################################################################################################
###############################################################################################################


nll.Gauss.cens<-function(init.val,datU,dist,thresh,sum=TRUE,print=FALSE)
{
  par<-init.val
  if(par[2]<0.001||par[2]>1.9999){return(10e10)}
  # transform data and threshold from uniform margins on to Gaussian
  z<-matrix(sapply(datU,qnorm),ncol=ncol(datU),nrow=nrow(datU))
  uz<-qnorm(thresh)

  if(length(uz)==1)  {
    uz<-rep(uz,dim(z)[2])
  } else if(length(uz)<dim(z)[2]){
    stop("Invalid censoring threshold")
  }

  Sig<-exp(-(dist/exp(par[1]))^par[2])

  if (!exists(".Random.seed", mode="numeric", envir=globalenv()))
    sample(NA)
  oldSeed <- get(".Random.seed", mode="numeric", envir=globalenv())

  # Establish censoring regimes (none, partial, full)

  tmp<-apply(z,1,function(t){(sum(t>uz))})
  ind.part.cens<-c(1:dim(z)[1])[tmp>0&tmp<dim(z)[2]]
  ind.full.cens<-c(1:dim(z)[1])[tmp==0]
  ind.no.cens<-c(1:dim(z)[1])[tmp==dim(z)[2]]

  # uncensored contribution
  if(length(ind.no.cens)>0)
  {
    if(sum){nll1<--sum(apply(matrix(z[ind.no.cens,],nrow=length(ind.no.cens)),1,dmvnorm,sigma=Sig,log=TRUE))+sum(dnorm(z[ind.no.cens,],log=TRUE))
    } else{nll1<--apply(matrix(z[ind.no.cens,],nrow=length(ind.no.cens)),1,dmvnorm,sigma=Sig,log=TRUE)+apply(dnorm(z[ind.no.cens,],log=TRUE),1,sum)}

  } else{nll1<-0}

  # partially censored contribution

  ll2<-NULL
  i<-1
  for(j in ind.part.cens)
  {
    cens<-which(z[j,]<=uz)
    nocens<-which(z[j,]>uz)

    Sig11<-Sig[cens,cens]-Sig[cens,nocens]%*%(solve(Sig[nocens,nocens])%*%Sig[nocens,cens])
    Sig11<-as.matrix(Sig11)
    if(!isSymmetric.matrix(Sig11))
    {
      Sig11<-(Sig11+t(Sig11))/2
    }
    mu11<-c(Sig[cens,nocens]%*%(solve(Sig[nocens,nocens])%*%z[j,nocens]))

    set.seed(123) # set seed to stabilize likelihood evaluation
    ll2[i]<-dmvnorm(z[j,nocens],sigma=as.matrix(Sig[nocens,nocens]),log=TRUE) + log(pmvnorm(upper=uz[cens],mean=mu11,sigma=Sig11)[1]) - sum(dnorm(z[j,nocens],log=TRUE))
    i<-i+1
  }
  if(sum){nll2<--sum(ll2)
  } else{nll2<- -ll2}

  # fully censored contribution
  set.seed(123)  # set seed to stabilize likelihood evaluation
  if(sum){nll3<--length(ind.full.cens)*log(pmvnorm(upper=uz,sigma=Sig)[1])
  } else{nll3<- -log(pmvnorm(upper=uz,sigma=Sig)[1])}
  #print(nll3)

  #restore random seed to its previous value
  assign(".Random.seed", oldSeed, envir=globalenv())

  if(sum){
    if(print){print(c(par,nll1+nll2+nll3))}
    return(nll1+nll2+nll3)
  } else{
      nll<-rep(nll3,dim(datU)[1])
      nll[ind.no.cens]<-nll1
      nll[ind.part.cens]<-nll2
      return(nll)
    }
}


nll.Gauss.cens.QMC<-function(init.val,datU,dist,thresh,sum=TRUE,print=FALSE,prime)
{
  par<-init.val
  if(par[2]<0.001||par[2]>1.9999){return(10e10)}
  # transform data and threshold from uniform margins on to Gaussian
  z<-matrix(sapply(datU,qnorm),ncol=ncol(datU),nrow=nrow(datU))
  uz<-qnorm(thresh)

  if(length(uz)==1)  {
    uz<-rep(uz,dim(z)[2])
  } else if(length(uz)<dim(z)[2]){
    stop("Invalid censoring threshold")
  }

  Sig<-exp(-(dist/exp(par[1]))^par[2])

  if (!exists(".Random.seed", mode="numeric", envir=globalenv()))
    sample(NA)
  oldSeed <- get(".Random.seed", mode="numeric", envir=globalenv())
  set.seed(123456)
  # Establish censoring regimes (none, partial, full)

  tmp<-apply(z,1,function(t){(sum(t>uz))})
  ind.part.cens<-c(1:dim(z)[1])[tmp>0&tmp<dim(z)[2]]
  ind.full.cens<-c(1:dim(z)[1])[tmp==0]
  ind.no.cens<-c(1:dim(z)[1])[tmp==dim(z)[2]]

  # uncensored contribution
  if(length(ind.no.cens)>0)
  {
    if(sum){nll1<--sum(apply(matrix(z[ind.no.cens,],nrow=length(ind.no.cens)),1,dmvnorm,sigma=Sig,log=TRUE))+sum(dnorm(z[ind.no.cens,],log=TRUE))
    } else{nll1<--apply(matrix(z[ind.no.cens,],nrow=length(ind.no.cens)),1,dmvnorm,sigma=Sig,log=TRUE)+apply(dnorm(z[ind.no.cens,],log=TRUE),1,sum)}

  } else{nll1<-0}

  # partially censored contribution

  ll2<-NULL
  i<-1
  seed<-sample(1:10000,size=length(ind.part.cens))
  seed2<-sample(1:10000,size=1) # for distribution function

  for(j in ind.part.cens)
  {
    cens<-which(z[j,]<=uz)
    nocens<-which(z[j,]>uz)

    Sig11<-Sig[cens,cens]-Sig[cens,nocens]%*%(solve(Sig[nocens,nocens])%*%Sig[nocens,cens])
    Sig11<-as.matrix(Sig11)
    if(!isSymmetric.matrix(Sig11))
    {
      Sig11<-(Sig11+t(Sig11))/2
    }
    mu11<-c(Sig[cens,nocens]%*%(solve(Sig[nocens,nocens])%*%z[j,nocens]))
    if(length(cens)>1){
      vec <- genVecQMC(prime, length(cens))
      set.seed(seed = seed[i])
      #set.seed(123) # set seed to stabilize likelihood evaluation
      ll2[i]<-dmvnorm(z[j,nocens],sigma=as.matrix(Sig[nocens,nocens]),log=TRUE) + log(mvPot::mvtNormQuasiMonteCarlo(upper=uz[cens]-mu11,cov=Sig11,p=vec$primeP,genVec=vec$genVec)[1]) - sum(dnorm(z[j,nocens],log=TRUE))
    } else{
      ll2[i]<-dmvnorm(z[j,nocens],sigma=as.matrix(Sig[nocens,nocens]),log=TRUE) + log(pnorm(uz[cens]-mu11,sd=sqrt(Sig11))) - sum(dnorm(z[j,nocens],log=TRUE))
    }
    i<-i+1
  }
  if(sum){nll2<--sum(ll2)
  } else{nll2<- -ll2}

  # fully censored contribution
  vec <- genVecQMC(prime, length(uz))

  set.seed(seed = seed2)
  #  set.seed(123)  # set seed to stabilize likelihood evaluation
  if(sum){nll3<--length(ind.full.cens)*log(mvPot::mvtNormQuasiMonteCarlo(upper=uz,cov=Sig,p=vec$primeP,genVec=vec$genVec)[1])
  } else{nll3<- -log(mvPot::mvtNormQuasiMonteCarlo(upper=uz,cov=Sig)[1],p=vec$primeP,genVec=vec$genVec)}
  #print(nll3)

  #restore random seed to its previous value
  assign(".Random.seed", oldSeed, envir=globalenv())

  if(sum){
    if(print){print(c(par,nll1+nll2+nll3))}
    return(nll1+nll2+nll3)
  } else{
    nll<-rep(nll3,dim(datU)[1])
    nll[ind.no.cens]<-nll1
    nll[ind.part.cens]<-nll2
    return(nll)
  }
}
###############################################################################################################
###############################################################################################################

#' Fit the Gaussian copula above a threshold via censored likelihood
#'
#' @param datU  n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
#' @param coord D*2 matrix of spatial coordinates
#' @param thresh quantile at which to censor, for example 0.95
#' @param model character vector specifying model for the correlation matrix of the Gaussian process W. AT THIS POINT ONLY THE POWERED EXPONENTIAL MODEL IS IMPLEMENTED
#' @param init.val gives the initial values for all parameters: first the parameter delta of R, and second the parameters of the model for the correlation function
#' @param fixed (must be same length as init) sets some parameters fixed to their initial values. This is used to optimize the likelihood function only with respect to some of the parameters.
#' @param std.err logical. If TRUE, optim will return the hessian
#' @param optim logical. If optim=FALSE, the function returns nllik(init.val) (no optimization). Can be useful to evaluate the calculation time for likelihood evaluation
#' @param method character vector specifying optimization method for optim
#' @param sum logical - if TRUE returns negative log-likelihood for sample; if FALSE then the value of the likelihood for each vector is returned. Not intended to be called by the user.
#' @param ... additional entries for the function optim
#' @return list with objects "mle", "nllik", "convergence", "hessian" and "counts"
#' @author Raphael Huser and Jenny Wadsworth
#' @export


fit.Gauss.Ddim<-function(datU, coord, thresh, model = "pexp", init.val, fixed = rep(FALSE, 2),
                         std.err = FALSE, optim = TRUE, method = "Nelder-Mead",
                         sum = TRUE, print=FALSE,...)
{
  if(optim){
  opt<-optim(nll.Gauss.cens,par=init.val,datU=datU,dist=rdist(coord),thresh=thresh,hessian=std.err,sum=TRUE,print=print,...)

  mle <- c()
  mle[!fixed] <- opt$par
  mle[fixed] <- init.val[fixed]
  z <- list()
  z$mle <- mle
  z$nllik <- opt$val
  z$convergence <- opt$convergence
  z$hessian <- opt$hessian
  z$counts <- opt$counts
  return(z)
  }else
  {
    return(nll.Gauss.cens(init.val=init.val,datU=datU,dist=rdist(coord),thresh=thresh,sum=sum))
  }
}


#' Fit the Gaussian copula above a threshold via censored likelihood. Uses faster QMC computation than fit.Gauss.Ddim.
#'
#' @param datU  n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
#' @param coord D*2 matrix of spatial coordinates
#' @param thresh quantile at which to censor, for example 0.95
#' @param model character vector specifying model for the correlation matrix of the Gaussian process W. AT THIS POINT ONLY THE POWERED EXPONENTIAL MODEL IS IMPLEMENTED
#' @param init.val gives the initial values for all parameters: first the parameter delta of R, and second the parameters of the model for the correlation function
#' @param fixed (must be same length as init) sets some parameters fixed to their initial values. This is used to optimize the likelihood function only with respect to some of the parameters.
#' @param std.err logical. If TRUE, optim will return the hessian
#' @param optim logical. If optim=FALSE, the function returns nllik(init.val) (no optimization). Can be useful to evaluate the calculation time for likelihood evaluation
#' @param method character vector specifying optimization method for optim
#' @param sum logical - if TRUE returns negative log-likelihood for sample; if FALSE then the value of the likelihood for each vector is returned. Not intended to be called by the user.
#' @param ... additional entries for the function optim
#' @param prime prime number for the QMC multivariate normal calculation
#' @return list with objects "mle", "nllik", "convergence", "hessian" and "counts"
#' @author Raphael Huser and Jenny Wadsworth
#' @export

fit.Gauss.Ddim.QMC<-function(datU, coord, thresh, model = "pexp", init.val, fixed = rep(FALSE, 2),
                             std.err = FALSE, optim = TRUE, method = "Nelder-Mead",
                             sum = TRUE, print=FALSE,prime=127,...)
{
  if(optim){
    opt<-optim(nll.Gauss.cens.QMC,par=init.val,datU=datU,dist=rdist(coord),thresh=thresh,hessian=std.err,sum=TRUE,print=print,prime=prime,...)

    mle <- c()
    mle[!fixed] <- opt$par
    mle[fixed] <- init.val[fixed]
    z <- list()
    z$mle <- mle
    z$nllik <- opt$val
    z$convergence <- opt$convergence
    z$hessian <- opt$hessian
    z$counts <- opt$counts
    return(z)
  }else
  {
    return(nll.Gauss.cens.QMC(init.val=init.val,datU=datU,dist=rdist(coord),thresh=thresh,sum=sum))
  }
}
###############################################################################################################
###############################################################################################################

#' calculates model-based chi_u in D dimensions, for the Gaussian copula with powered exponential covariance function
#' @param  u sequence of values at which to calculate chi
#' @param cov.par scale and shape parameter (respectively) of the powered exponential covariance function
#' @param dist distance matrix between sites
#'
#' @return returns a vector corresponding to the value of chiu at each value of u
#' @author Raphael Huser and Jenny Wadsworth
#'
#' @export


chiu.Gauss<-function (u, cov.par, dist)
{
  sigma <- exp(-(dist/cov.par[1])^cov.par[2])
  chis <- rep(NA, length(u))
  for (i in 1:length(u)) {
    if (u[i] == 1) {
      try(chis[i] <- 0)
    }
    else {
      try(chis[i] <- pmvnorm(lower=rep(qnorm(u[i]),dim(sigma)[1]),sigma=sigma)/(1 - u[i]))
    }
  }
  return(chis)
}



###############################################################################################################
###############################################################################################################


unif<-function(x){rank(x)/(length(x)+1)}


###############################################################################################################
###############################################################################################################

MLE.Chi<-function(x, u, k, q1, q2=1)
{
  if(is.null(u)){
    u<-seq(q1,q2,len=k+1)
    u<-u[-(k+1)]
  }
  k<-length(u)
  x.unif<-x
  for(j in 1:dim(x)[2])
  {
    x.unif[,j]<-unif(x[,j])
  }

  d<-dim(x.unif)[2]
  x.umin<-apply(x.unif,1,min)
  thetahat<-NULL
  for(i in 1:k)
  {
    thetahat[i]<-mean(x.umin>=u[i])/(1-u[i])
  }
  return(list(mle=thetahat))
}

MLE.Eta<-function(x, u, k, q1, q2=1)
{
  if(is.null(u)){
    u<-seq(q1,q2,len=k+1)
    u<-u[-(k+1)]
  }
  k<-length(u)

  x.unif<-x
  for(j in 1:dim(x)[2])
  {
    x.unif[,j]<-unif(x[,j])
  }

  d<-dim(x.unif)[2]
  x.umin<-apply(x.unif,1,min)
  etahat<-NULL
  for(i in 1:k)
  {
    etahat[i]<-mean(-log(1-x.umin[x.umin>=u[i]])+log(1-u[i]))
  }
  return(list(mle=etahat))
}



###############################################################################################################
###############################################################################################################

#' empirical estimation of chi_u function, with (block) bootstrap confidence intervals
#'
#' @param x data in an n by D matrix. Any marginals is fine: a nonparametric transformation will be applied.
#' @param u optionally a sequence of values at which to estimate chi
#' @param k if u is not supplied, k is the number of points at which to estimate chi
#' @param q1 if u is not supplied, q1 is the lowest quantile at which chi will be estimated
#' @param q2 if u is not supplied, q2 set the upper limit of chi estimation, however, the true upper limit will be that given by seq(q1,q2,len=k+1)[-(k+1)]
#' @param pmar margins for plotting
#' @param rtn logical; if TRUE will output estimates and bootstrap sampling distribution. Else will only plot estimate.
#' @param nbs number of bootstrap samples for creation of confidence intervals
#' @param blocklength length of block. If set equal to 1 then standard bootstrap; else this is a (non-overlapping) block bootstrap
#'
#' @return plots the estimate and approx 95\% CI. If rtn=TRUE then a list with objects chi (estimate of chi_u at the sequence u), chiBS (bootstrap sample estimates of the same), eta (estimate of the coefficient of tail dependence, eta above the thresholds defined by u), etaBS (bootstrap sample distribtuion of the same)
#' @author Jenny Wadsworth
#'
#' @export


chiu.est<-function(x, u=NULL, k, q1, q2=1, pmar=c(5.5,7,3,3), rtn=FALSE, nbs, blocklength,...)
{
  par(mfrow=c(1, 1))
  par(mar=pmar)

  par(las=1)


  if(is.null(u)){
    u<-seq(q1,q2,len=k+1)
    u<-u[-(k+1)]
    }
  k<-length(u)

  M1<-MLE.Chi(x=x, u=u, k=k, q1=q1, q2=q2)
  ME1<-MLE.Eta(x=x, u=u, k=k, q1=q1, q2=q2)

  Mbs<-matrix(0,ncol=k,nrow=nbs)
  MEbs<-matrix(0,ncol=k,nrow=nbs)


  for(i in 1:nbs)
  {
    cat(paste(rep("\b",15),collapse=""))
    cat(paste(c(round(i/nbs,2))*100, "%"))

    if(blocklength==1)
    {
      xbs<-x[sample(1:dim(x)[1],replace=T),]
    }
    else{
    n<-floor(dim(x)[1]/blocklength)
    n.bs<-sample(1:n,size=n,replace=T)
    xbs<-NULL
    for(s in 1:n)
    {
      xbs<-rbind(xbs,x[(blocklength*(n.bs[s]-1)+1):(blocklength*n.bs[s]),])
    }
    }
    Mbs[i,]<-MLE.Chi(x=xbs, u=u, k=k, q1=q1, q2=q2)$mle
    MEbs[i,]<-MLE.Eta(x=xbs, u=u, k=k, q1=q1, q2=q2)$mle
    print(i)
  }

  CI<-cbind(apply(Mbs,2,quantile,0.025),apply(Mbs,2,quantile,0.975))

  plot(u,M1$mle,ylim=c(min(CI[,1]),max(CI[,2])),...)
  lines(u,CI[,1], lty=2)
  lines(u,CI[,2], lty=2)

  if(rtn)
  {
    return(list(chi=M1$mle,chiBS=Mbs,eta=ME1,etaBS=MEbs))
  }
}
