# Code for the bivariate copula model defined by X=R^(d) * W^(1-d)
# These functions work on the scale Y=log(X)
########################################################################

# QF.y
# Quantile function to return marginal quantiles on Y scale from uniform scale

# Arguments:
# u - scalar between (0,1)
# d - value of delta

QF.y<-function(u,d)
{
  dummy.d<-function(y)
  {
    x<-exp(y)
    (1-u) -(d/(2*d-1))*x^(-1/d) + ((1-d)/(2*d-1))*x^(-1/(1-d))
  }
  if(abs(d-0.5)>0.00001)
  {
    ur<-uniroot(dummy.d,interval=c(0,100),extendInt = "yes")
  }
  else{
    dummy.0.5<-function(y)
    {
      x<-exp(y)
      (1-u)-x^(-2)*(2*log(x)+1)
    }
    ur<-uniroot(dummy.0.5,interval=c(0,100),extendInt = "yes")
  }
  return(ur$root)
}

##########################################################################
# f.m.y
# marginal density function

# Arguments:
# y - scalar between (0,Inf)
# d - value of delta

f.m.y<-function(y,d)
{
  x<-exp(y)
  if(abs(d-0.5)>0.0001)
  {
    dens<-((1/(2*d-1))*x^(-1/d-1) - (1/(2*d-1))*x^(-1/(1-d)-1))*exp(y)
    return(dens)
  }
  else{return((x^(-3)*(4*log(x)))*exp(y))}
}


##########################################################################

# F.m.y
# marginal distribution function

# Arguments:
# y - scalar between (0,Inf)
# a - value of alpha

F.m.y<-function(y,d)
{
  x<-exp(y)
  if(abs(d-0.5)>0.0001)
  {
    1-((d/(2*d-1))*x^(-1/d) - ((1-d)/(2*d-1))*x^(-1/(1-d)))
  }
  else{return(1-(x^(-2)*(2*log(x)+1)))}
}


##########################################################################

# Fbar.m.y
# marginal survivor function

# Arguments:
# y - scalar between (0,Inf)
# a - value of alpha

Fbar.m.y<-function(y,d)
{
  x<-exp(y)
  if(abs(d-0.5)>0.0001)
  {
    (d/(2*d-1))*x^(-1/d) - ((1-d)/(2*d-1))*x^(-1/(1-d))
  }
  else{return((x^(-2)*(2*log(x)+1)))}
}


##########################################################################
##########################################################################
# Functions for different W models (on V=logW exponential scale)

# k = V(1/x) is an "inverted exponent function" for inverted extreme value distribution.
# k1, k2 and k12 are partial derivatives
# For now, only the inverted logistic model is implemented, but it should be simple to extend to something asymmetric

# Arguments
# v - scalar between (0,Inf)
# theta - logistic parameter in the range (0,1]

kl<-function(v,theta)
{
  (v[1]^(1/theta)+v[2]^(1/theta))^theta
}

kl1<-function(v,theta)
{
  if(v[1]==0&v[2]==0){return(0)}
  else{
    v[1]^(1/theta-1)*(v[1]^(1/theta)+v[2]^(1/theta))^(theta-1)
  }
}

kl2<-function(v,theta)
{
  if(v[1]==0&v[2]==0){return(0)}
  else{
    v[2]^(1/theta-1)*(v[1]^(1/theta)+v[2]^(1/theta))^(theta-1)
  }
}

kl12<-function(v,theta)
{
  if(v[1]==0&v[2]==0){return(0)}
  else{
    ((theta-1)/theta)*v[1]^(1/theta-1)* v[2]^(1/theta-1)*(v[1]^(1/theta)+v[2]^(1/theta))^(theta-2)
  }
}

##########################################################################

kct<-function(v,theta)
{
  q<-theta[1]*v[2]/(theta[1]*v[2]+theta[2]*v[1])
  return(v[1]*(1-pbeta(q,theta[1]+1,theta[2])) +v[2]*pbeta(q,theta[1],theta[2]+1))
}


kct1<-function(v,theta)
{
  q<-theta[1]*v[2]/(theta[1]*v[2]+theta[2]*v[1])

  bit1<-1-pbeta(q,theta[1]+1,theta[2])
  bit2<-prod(theta)*prod(v)*dbeta(q,theta[1]+1,theta[2]) - prod(theta)*v[2]^2*dbeta(q,theta[1],theta[2]+1)
  return(bit1+bit2)
}

kct2<-function(v,theta)
{
  q<-theta[1]*v[2]/(theta[1]*v[2]+theta[2]*v[1])

  bit1<-pbeta(q,theta[1],theta[2]+1)
  bit2<-prod(theta)*prod(v)*dbeta(q,theta[1],theta[2]+1) - prod(theta)*v[1]^2*dbeta(q,theta[1]+1,theta[2])
  return(bit1+bit2)
}


kct12<-function(v,theta)
{
  q<-theta[1]*v[2]/(theta[1]*v[2]+theta[2]*v[1])
  bit1<- -(v[1]+v[2])^(-3)*(v[1]*v[2])*hbvct(v[2]/(v[2]+v[1]),theta[1],theta[2])
  return(bit1)
}


##########################################################################

# f.jt.v.IEVL/f.jt.v.IEVct
# exponential scale density for inverted extreme value (logistic / ct-dirichlet)

# Arguments
# v - scalar between (0,Inf)
# theta - logistic parameter in the range (0,1] OR ct-dirichlet parameter in (0,Inf)^2

f.jt.v.IEVL<-function(v,theta)
{
  dens<- (kl1(v, theta=theta)*kl2(v, theta=theta) - kl12(v, theta=theta))*exp(-kl(v,theta=theta))
  return(dens)
}

f.jt.v.IEVct<-function(v,theta)
{
  dens<- (kct1(v, theta=theta)*kct2(v, theta=theta) - kct12(v, theta=theta))*exp(-kct(v,theta=theta))
  return(dens)
}

##########################################################################

# f.jt.v.Gauss
# exponential scale density for Gaussian

# Arguments
# v - scalar between (0,Inf)
# theta - correlation parameter in the range (-1,1)

f.jt.v.Gauss<-function(v,theta)
{
  sigma<-matrix(c(1,theta,theta,1),2,2)
  z<-qnorm(1-exp(-v))

  # Use symmetry of Gaussian if exp(-v) very small and above resolves to infinity
  if(any(z==Inf))
  {
    z<--qnorm(exp(-v))
  }
  dens<-mvtnorm::dmvnorm(z,sigma=sigma)*prod(exp(-v)/dnorm(z))
  return(dens)
}


##########################################################################

# F.jt.v.IEVL/ F.jt.v.IEVct
# exponential scale joint distribution function for inverted extreme value

# Arguments
# v - vector in (0,Inf)^2
# theta - logistic parameter in the range (0,1] OR ct-dirichlet parameter in (0,Inf)^2

F.jt.v.IEVL<-function(v,theta)
{
  1-exp(-v[1])-exp(-v[2])+exp(-kl(v,theta=theta))
}

F.jt.v.IEVct<-function(v,theta)
{
  1-exp(-v[1])-exp(-v[2])+exp(-kct(v,theta=theta))
}

##########################################################################

# F.jt.v.Gauss
# exponential scale joint distribution function for Gaussian

# Arguments
# v - scalar between (0,Inf)
# theta - correlation parameter in the range (-1,1)

F.jt.v.Gauss<-function(v,theta)
{
  sigma<-matrix(c(1,theta,theta,1),2,2)
  mvtnorm::pmvnorm(upper=qnorm(1-exp(-v)),sigma=sigma)
}


##########################################################################

# Fpartial.jt.v.IEVL / Fpartial.jt.v.IEVct
# exponential partial derivative of distribution function for inverted extreme value

# Arguments
# v - vector in (0,Inf)^2
# theta - logistic parameter in the range (0,1] OR ct-dirichlet parameter in (0,Inf)^2

F.partial.jt.v.IEVL<-function(v,theta,which)
{
  notwhich<-c(1,2)[-which]
  return(exp(-v[which])-kl1(c(v[which],v[notwhich]),theta=theta)*exp(-kl(v,theta=theta)))
}

F.partial.jt.v.IEVct<-function(v,theta,which)
{
  if(which==1){return(exp(-v[1])-kct1(c(v[1],v[2]),theta=theta)*exp(-kct(v,theta=theta)))}
  else if(which==2){return(exp(-v[2])-kct2(c(v[1],v[2]),theta=theta)*exp(-kct(v,theta=theta)))}
}


##########################################################################

# F.partial.jt.v.Gauss
# exponential scale partial derivative of distribution function for Gaussian

# Arguments
# v - scalar between (0,Inf)
# theta - correlation parameter in the range (-1,1)

F.partial.jt.v.Gauss<-function(v,theta,which)
{
  notwhich<-c(1,2)[-which]
  a<-qnorm(1-exp(-v))
  return(pnorm((a[notwhich]-theta*a[which])/sqrt(1-theta^2))*pnorm(-a[which]))
}


##########################################################################
##########################################################################

# F.jt.y
# joint distribution function

# Arguments
# y - vector in (0,Inf)^2
# d - value of delta
# theta - logistic parameter OR Gaussian correlation
# model - character


F.jt.y<-function(y,d,theta,model)
{
  if(model=="IEVL"){
    F.jt.v<-F.jt.v.IEVL
  } else if(model=="IEVct"){
    F.jt.v<-F.jt.v.IEVct
  } else if(model=="Gauss"){F.jt.v<-F.jt.v.Gauss}

  x<-exp(y)

  if(model=="IEVL"&&(y[1]==y[2]))
  {
    eta<-1/kl(c(1,1),theta=theta)
    d1<-(1-d)*eta
    jsurv<-(d/(d-d1))*x[1]^(-1/d) - (d1/(d-d1))*x[1]^(-1/d1)
    return(1-2*Fbar.m.y(y[1],d=d)+jsurv)
  }
  if(model=="IEVct"&&(y[1]==y[2]))
  {
    eta<-1/kct(c(1,1),theta=theta)
    d1<-(1-d)*eta
    jsurv<-(d/(d-d1))*x[1]^(-1/d) - (d1/(d-d1))*x[1]^(-1/d1)
    return(1-2*Fbar.m.y(y[1],d=d)+jsurv)
  }

  to.int<-function(v)
  {
    s<-c(y[1]-d*v, y[2]-d*v)/(1-d)
    intgrd<-F.jt.v(s,theta=theta)*exp(-v)
    return(intgrd)
  }
  to.int.vec<-Vectorize(to.int)
  int<-integrate(to.int.vec,lower=0,upper=min(y)/d,abs.tol=0)
  ans<-int$value

  return(ans)
}



##########################################################################

# F.jt.y
# joint distribution function

# Arguments
# y - vector in (0,Inf)^2
# d - value of delta
# theta - logistic parameter OR Gaussian correlation
# model - character


F.partial.jt.y<-function(y,d,theta,model,which)
{
  if(model=="IEVL"){
    F.partial.jt.v<-F.partial.jt.v.IEVL
  } else if(model=="IEVct"){
    F.partial.jt.v<-F.partial.jt.v.IEVct
  }else if(model=="Gauss"){F.partial.jt.v<-F.partial.jt.v.Gauss}

  to.int<-function(v)
  {
    s<-c(y[1]-d*v, y[2]-d*v)/(1-d)
    intgrd<-F.partial.jt.v(s,theta=theta,which=which)*exp(-v)
    return(intgrd)
  }
  to.int.vec<-Vectorize(to.int)
  int<-integrate(to.int.vec,lower=0,upper=min(y)/d,abs.tol=0)
  ans<-int$value/(1-d)

  return(ans)
}



##########################################################################
Fbar.jt.y<-function(y,d,theta,model)
{
  1-F.m.y(y[1],d=d)-F.m.y(y[2],d=d)+F.jt.y(y=y, d=d, theta=theta, model=model)
}
##########################################################################

# f.jt.y
# joint density function

# Arguments
# y - vector in (0,Inf)^2
# a - value of alpha
# theta - logistic parameter OR Gaussian correlation
# model - character


f.jt.y<-function(y, d, theta, model)
{
  if(model=="IEVL"){
    f.jt.v<-f.jt.v.IEVL
  } else if(model=="IEVct"){
    f.jt.v<-f.jt.v.IEVct
  } else if(model=="Gauss"){f.jt.v<-f.jt.v.Gauss}

  to.int<-function(v)
  {
    s<-c(y[1]-d*v, y[2]-d*v)/(1-d)
    out<-f.jt.v(v=s,theta=theta)*exp(-v)
    return(out)
  }
  to.int.vec<-Vectorize(to.int)

  int<-integrate(to.int.vec,lower=0,upper=min(y)/d, abs.tol=0)
  dens<-int$val *(1-d)^(-2)
  return(dens)
}



##########################################################################
# cop.dens.y
# copula density


# Arguments:
# u - n * 2 matrix of data on uniform margins
# a - value of alpha
# theta - value of parameter for W model
# model - character "IEVL" for inverted extreme value logistic; "Gauss" for BV normal


cop.dens.y<-function(u,d,theta,model=model)
{
  y<-NULL
  y[1]<-QF.y(u=u[1],d=d)
  y[2]<-QF.y(u=u[2],d=d)

  num<-f.jt.y(y=y,d=d,theta=theta,model=model)
  den<-f.m.y(y[1],d=d)*f.m.y(y[2],d=d)
  return(num/den)
}


##########################################################################
# nll.cop.y
# negative log likelihood (uncensored)

# Arguments:
# par - parameter vector (2-dim at the moment) of (alpha, theta), where theta is the (currently scalar) parameter for the W model
# u - n * 2 matrix of data on uniform margins
# model - character "IEVL" for inverted extreme value logistic; "Gauss" for BV normal

nll.cop.y<-function(par,u,model)
{
  d<-par[1]
  theta<-par[2]

  dens<-apply(u,1,cop.dens.y,d=d,theta=theta,model=model)
  nll<--sum(log(dens))
  print(nll)
  return(nll)
}

# determine exceedances

exc.logic<-function(y,thresh)
{
  y[1]>thresh[1]||y[2]>thresh[2]
}


##########################################################################
# nll.cens.y
# negative log likelihood (censored)

# Arguments:
# par - parameter vector (2-dim at the moment) of (alpha, theta), where theta is the (currently scalar) parameter for the W model
# u - n * 2 matrix of data on uniform margins
# thresh - bivariate censoring threshold in (0,1)^2
# model - character "IEVL" for inverted extreme value logistic; "Gauss" for BV normal
# alim.lower, alim.upper - lower and upper limits for alpha; to avoid numerical issues if necessary

nll.cens.y<-function(par,u,thresh,model,dlim.lower=0.02,dlim.upper=0.98,print=FALSE)
{
  d<-par[1]
  theta<-par[2:length(par)]

  if(length(thresh)==1){thresh<-rep(thresh,2)}

  if(model=="IEVL"){theta.out<-theta<0.001 || theta>1}
  else if(model=="IEVct"){theta.out<- theta[1]< 0.001 || theta[2]<0.001}
  else if(model=="Gauss"){theta.out<- theta< -0.999 || theta>0.999}

  # keep a away from 0 or 1 due to numerical issues; still gives wide range of dependence
  if(d< dlim.lower || d> dlim.upper || theta.out){return(10e10)}
  exc<-apply(u,1,exc.logic,thresh=thresh)

  u.exc<-u[exc,]
  u.exc<-matrix(u.exc,ncol=2)

  n.n.exc<-dim(u)[1]-sum(exc)

  # Censored bit

  thresh.y<-NULL
  thresh.y[1]<-QF.y(u=thresh[1],d=d)
  thresh.y[2]<-QF.y(u=thresh[2],d=d)

  p<-F.jt.y(thresh.y,d=d,theta=theta,model=model)
  cens<-n.n.exc*log(p)

  # Uncensored bit
  tmp<-tryCatch(apply(u.exc, 1, cop.dens.y, d=d, theta=theta,model=model),error=function(e) e)
  if(is.element("error", class(tmp)))
  {
    warning("Numerical difficulties (probably an integration error) occurred at parameter values:", round(d,4),
            ",", round(theta,4), "\n NA returned to optimization")
    return(NA)
  }
  uncens<-sum(log(tmp))

  nll<--cens-uncens
  if(print){print(c(par,nll))}
  return(nll)
}


##########################################################################
# nll.cens.y
# negative log likelihood (censored) - allowing use of multiple cores with multicore package

# Arguments:
# par - parameter vector (2-dim at the moment) of (alpha, theta), where theta is the (currently scalar) parameter for the W model
# u - n * 2 matrix of data on uniform margins
# thresh - bivariate censoring threshold in (0,1)^2
# model - character "IEVL" for inverted extreme value logistic; "Gauss" for BV normal
# ncore - number of cores to use

nll.cens.y.mc<-function(par,u,thresh,model,dlim.lower=0.02,dlim.upper=0.98,ncore=4,print=FALSE)
{
  d<-par[1]
  theta<-par[2:length(par)]

  if(length(thresh)==1){thresh<-rep(thresh,2)}

  if(model=="IEVL"){theta.out<-theta<0.001 || theta>1}
  else if(model=="IEVct"){theta.out<- theta[1]< 0.001 || theta[2]<0.001}
  else if(model=="Gauss"){theta.out<- theta< -0.999 || theta>0.999}

  # keep a away from 0 or 1 due to numerical issues; still gives wide range of dependence
  if(d< dlim.lower || d> dlim.upper || theta.out){return(10e10)}
  exc<-apply(u,1,exc.logic,thresh=thresh)

  u.exc<-u[exc,]
  u.exc<-matrix(u.exc,ncol=2)

  n.n.exc<-dim(u)[1]-sum(exc)

  # Censored bit

  thresh.y<-NULL
  thresh.y[1]<-QF.y(u=thresh[1],d=d)
  thresh.y[2]<-QF.y(u=thresh[2],d=d)

  p<-F.jt.y(thresh.y,d=d,theta=theta,model=model)
  cens<-n.n.exc*log(p)

  # Uncensored bit
  u.exc.list<-list()
  for(i in 1:sum(exc))
  {
    u.exc.list[[i]]<-u.exc[i,]
  }

  suppressWarnings(tmp<-tryCatch(mclapply(u.exc.list, cop.dens.y, d=d, theta=theta,model=model,mc.cores=ncore), error=function(e) e))
  if(is.element("try-error", sapply(tmp,class)))
  {
    warning("Numerical difficulties (probably an integration error) occurred at parameter values:", round(d,4),
            ",", round(theta,4), "\n NA returned to optimization")
    return(NA)
  }

  uncens<-sum(log(unlist(tmp)))

  nll<--cens-uncens
  if(print){print(c(par,nll))}
  return(nll)
}



##########################################################################
# nll.cens.y2
# negative log likelihood (censored) - with partial and full censoring

# Arguments:
# par - parameter vector (2-dim at the moment) of (alpha, theta), where theta is the (currently scalar) parameter for the W model
# u - n * 2 matrix of data on uniform margins
# thresh - bivariate censoring threshold in (0,1)^2
# model - character "IEVL" for inverted extreme value logistic; "Gauss" for BV normal
# alim.lower, alim.upper - lower and upper limits for alpha; to avoid numerical issues if necessary


nll.cens.y2<-function(par,u,thresh,model,dlim.lower=0.02,dlim.upper=0.98,print=FALSE)
{
  d<-par[1]
  theta<-par[2:length(par)]

  if(length(thresh)==1){thresh<-rep(thresh,2)}

  if(model=="IEVL"){theta.out<-theta<0.001 || theta>1}
  else if(model=="IEVct"){theta.out<- theta[1]< 0.001 || theta[2]<0.001}
  else if(model=="Gauss"){theta.out<- theta< -0.999 || theta>0.999}

  # keep a away from 0 or 1 due to numerical issues; still gives wide range of dependence
  if(d< dlim.lower || d> dlim.upper || theta.out){return(10e10)}
  jt.exc<-apply(u,1,function(y,thresh){y[1]>thresh[1]&&y[2]>thresh[2]},thresh=thresh)
  exc1<-apply(u,1,function(y,thresh){y[1]>thresh[1]&&y[2]<=thresh[2]},thresh=thresh)
  exc2<-apply(u,1,function(y,thresh){y[1]<=thresh[1]&&y[2]>thresh[2]},thresh=thresh)

  u.jt.exc<-u[jt.exc,]
  u.jt.exc<-matrix(u.jt.exc,ncol=2)# force to be a matrix to avoid trouble later if only a single obs in this region
  u.exc1<-u[exc1,]
  u.exc1<-matrix(u.exc1,ncol=2)
  u.exc2<-u[exc2,]
  u.exc2<-matrix(u.exc2,ncol=2)

  n.n.exc<-dim(u)[1]-sum(jt.exc)-sum(exc1)-sum(exc2)

  # Fully censored bit

  thresh.y<-NULL
  thresh.y[1]<-QF.y(u=thresh[1],d=d)
  thresh.y[2]<-QF.y(u=thresh[2],d=d)

  p<-F.jt.y(thresh.y,d=d,theta=theta,model=model)
  cens<-n.n.exc*log(p)

  # Part censored bit
  # Exceedance in coord 1

  # (Firstly go from uniform scale to y scale)

  y.exc1<-matrix(0,ncol=2,nrow=sum(exc1))
  y.exc1[,1]<-sapply(u.exc1[,1],QF.y,d=d)
  y.exc1[,2]<-thresh.y[2]

  tmp<-tryCatch(apply(y.exc1,1,F.partial.jt.y,d=d,theta=theta,model=model,which=1), error=function(e) e)
  if(is.element("error", class(tmp)))
  {
    warning("Numerical difficulties (probably an integration error) occurred at parameter values:", round(d,4),
            ",", round(theta,4), "\n NA returned to optimization")
    return(NA)
  }

  # need to add in jacobian term
  pcens1<-sum(log(tmp))-sum(log(f.m.y(y.exc1[,1],d=d)))

  # Exceedance in coord 2

  y.exc2<-matrix(0,ncol=2,nrow=sum(exc2))
  y.exc2[,2]<-sapply(u.exc2[,2],QF.y,d=d)
  y.exc2[,1]<-thresh.y[1]

  tmp<-tryCatch(apply(y.exc2,1,F.partial.jt.y,d=d,theta=theta,model=model,which=2), error=function(e) e)
  if(is.element("error", class(tmp)))
  {
    warning("Numerical difficulties (probably an integration error) occurred at parameter values:", round(d,4),
            ",", round(theta,4), "\n NA returned to optimization")
    return(NA)
  }
  pcens2<-sum(log(tmp))-sum(log(f.m.y(y.exc2[,2],d=d)))

  # Uncensored bit
  tmp<-tryCatch(apply(u.jt.exc, 1, cop.dens.y, d=d, theta=theta,model=model), error=function(e) e)
  if(is.element("error", class(tmp)))
  {
    warning("Numerical difficulties (probably an integration error) occurred at parameter values:", round(d,4),
            ",", round(theta,4), "\n NA returned to optimization")
    return(NA)
  }
  uncens<-sum(log(tmp))

  nll<--cens-pcens1-pcens2-uncens
  if(print){print(c(par,nll))}
  return(nll)
}





nll.cens.y2.mc<-function(par,u,thresh,model,dlim.lower=0.02,dlim.upper=0.98,print=FALSE,ncore=2)
{
  d<-par[1]
  theta<-par[2:length(par)]

  if(length(thresh)==1){thresh<-rep(thresh,2)}

  if(model=="IEVL"){theta.out<-theta<0.001 || theta>1}
  else if(model=="IEVct"){theta.out<- theta[1]< 0.001 || theta[2]<0.001}
  else if(model=="Gauss"){theta.out<- theta< -0.999 || theta>0.999}

  # keep a away from 0 or 1 due to numerical issues; still gives wide range of dependence
  if(d< dlim.lower || d> dlim.upper || theta.out){return(10e10)}
  jt.exc<-apply(u,1,function(y,thresh){y[1]>thresh[1]&&y[2]>thresh[2]},thresh=thresh)
  exc1<-apply(u,1,function(y,thresh){y[1]>thresh[1]&&y[2]<=thresh[2]},thresh=thresh)
  exc2<-apply(u,1,function(y,thresh){y[1]<=thresh[1]&&y[2]>thresh[2]},thresh=thresh)

  u.jt.exc<-u[jt.exc,]
  u.jt.exc<-matrix(u.jt.exc,ncol=2)# force to be a matrix to avoid trouble later if only a single obs in this region
  u.exc1<-u[exc1,]
  u.exc1<-matrix(u.exc1,ncol=2)
  u.exc2<-u[exc2,]
  u.exc2<-matrix(u.exc2,ncol=2)

  n.n.exc<-dim(u)[1]-sum(jt.exc)-sum(exc1)-sum(exc2)

  # Fully censored bit

  thresh.y<-NULL
  thresh.y[1]<-QF.y(u=thresh[1],d=d)
  thresh.y[2]<-QF.y(u=thresh[2],d=d)

  p<-F.jt.y(thresh.y,d=d,theta=theta,model=model)
  cens<-n.n.exc*log(p)

  # Part censored bit
  # Exceedance in coord 1

  # (Firstly go from uniform scale to y scale)
  if(sum(exc1)>0)
  {
    y.exc1<-list(sum(exc1))
    y.exc1c1<-sapply(u.exc1[,1],QF.y,d=d)
    for(i in 1:sum(exc1))
    {
      y.exc1[[i]]<-c(y.exc1c1[i],thresh.y[2])
    }

    suppressWarnings(tmp<-tryCatch(mclapply(y.exc1,F.partial.jt.y,d=d,theta=theta,model=model,which=1,mc.cores=ncore), error=function(e) e))
    if(is.element("try-error", sapply(tmp,class)))
    {
      warning("Numerical difficulties (probably an integration error) occurred at parameter values:", round(d,4),
              ",", round(theta,4), "\n NA returned to optimization")
      return(NA)
    }
    pcens1<-sum(log(unlist(tmp)))-sum(log(f.m.y(y.exc1c1,d=d)))
  }  else{pcens1<-0}

  # Exceedance in coord 2

  if(sum(exc2)>0)
  {
    y.exc2<-list(sum(exc2))
    y.exc2c2<-sapply(u.exc2[,2],QF.y,d=d)
    for(i in 1:sum(exc2))
    {
      y.exc2[[i]]<-c(thresh.y[1],y.exc2c2[i])
    }
    suppressWarnings(tmp<-tryCatch(mclapply(y.exc2,F.partial.jt.y,d=d,theta=theta,model=model,which=2,mc.cores=ncore), error=function(e) e))
    if(is.element("try-error", sapply(tmp,class)))
    {
      warning("Numerical difficulties (probably an integration error) occurred at parameter values:", round(d,4),
              ",", round(theta,4), "\n NA returned to optimization")
      return(NA)
    }
    pcens2<-sum(log(unlist(tmp)))-sum(log(f.m.y(y.exc2c2,d=d)))
  } else{pcens2<-0}

  # Uncensored bit

  if(sum(jt.exc)>0)
  {
    u.exc.list<-list()
    for(i in 1:sum(jt.exc))
    {
      u.exc.list[[i]]<-u.jt.exc[i,]
    }
    suppressWarnings(tmp<-tryCatch(mclapply(u.exc.list, cop.dens.y, d=d, theta=theta,model=model,mc.cores=ncore), error=function(e) e))
    if(is.element("try-error", sapply(tmp,class)))
    {
      warning("Numerical difficulties (probably an integration error) occurred at parameter values:", round(d,4),
              ",", round(theta,4), "\n NA returned to optimization")
      return(NA)
    }
    uncens<-sum(log(unlist(tmp)))
  } else{uncens<-0}

  nll<--cens-pcens1-pcens2-uncens
  if(print){print(c(par,nll))}
  return(nll)
}

#############################################################################
# proflik.d
# create profile likelihood for delta

# Arguments
# arange - 2-vector giving upper and lower limits for calculating profile likelihood
# alength - number of points in the range at which to calculate the profile likelihood
# u - n*2 matrix of data on uniform margins
# thresh - bivariate cendoring threshold
# model - character "IEVL" for inverted extreme value logistic; "Gauss" for BV normal
# plot - logical, plot profile log-likelihood
# ncore - number of cores to use

proflik.d<-function(drange,dlength,u,thresh,model,plot=TRUE,print=TRUE,ncore=4)
{
  dseq<-seq(drange[1],drange[2],length=dlength)

  # keep a away from 0 or 1 due to numerical issues; still gives wide range of dependence

 if(ncore==1)
 {
   if(cens.type=="max"){nll<-function(theta,d){nll.cens.y(par = c(d,theta),model = model,u = u,thresh = thresh)}}
   else if(cens.type=="partial"){nll<-function(theta,d){nll.cens.y2(par = c(d,theta),model = model,u = u,thresh = thresh)}}
 }
  else if(ncore>1)
  {
    if(cens.type=="max"){nll<-function(theta,d){nll.cens.y.mc(par = c(d,theta),model = model,u = u,thresh = thresh, ncore=ncore)}}
    else if(cens.type=="partial"){nll<-function(theta,d){nll.cens.y2.mc(par = c(d,theta),model = model,u = u,thresh = thresh, ncore=ncore)}}
  }

  if(is.element(model,c("IEVL","Gauss")))
  {
  if(model=="IEVL"){theta.range<-c(0.001,0.999)}
  else if(model=="Gauss"){theta.range<-c(-0.999,0.999)}

  pl.d<-NULL
  for(i in 1:dlength)
  {
    pl.d[i]<-optimize(nll,interval = theta.range, d=dseq[i])$objective
    if(print){print(c(dseq[i],-pl.d[i]))}
  }
  }else{
    pl.d<-NULL
    for(i in 1:dlength)
    {
      pl.d[i]<-optim(nll,par=c(1,1),d=dseq[i])$value
      if(print){print(c(dseq[i],-pl.d[i]))}
    }
  }

  if(plot){
    plot(dseq,-pl.d,typ="l",xlab=expression(delta),ylab="Profile log-likelihood")
    abline(h=max(-pl.d)-qchisq(0.95,df=1)/2,col=2)
  }
  return(cbind(dseq,-pl.d))
}


##########################################################################
# rMar
# Simulate random variables with the marginal distribution R^(d)*W^(d), or on log (Y) scale

# Arguments:
# n - number of variables to simulate
# d - parameter delta
# log - logical; return on Y scale?

rMar<-function(n,d,log=T)
{
  R<-1/runif(n)
  W<-1/runif(n)
  Y<-d*log(R) + (1-d)*log(W)
  if(log){return(Y)}
  else{return(exp(Y))}
}

##########################################################################
# rCop
#' Simulate realizations of the bivariate copula R^(delta)*W^(1-delta) for different W

#' @param n  number of realizations to simulate
#' @param delta  parameter delta
#' @param theta  parameter(s) of W distribution
#' @param model  character "IEVL" for inverted extreme value logistic; "IEVct" for inverted Coles-Tawn Dirichlet; or "Gauss" for BV normal
#' @param scale  "unif" for uniform, "y" for Y=d*logR +(1-d)*logW or "x" for exp(y)
#' @author Jenny Wadsworth
#' @examples
#'rC2(n=1,delta=0.4,theta=-0.5,model="Gauss")
#'rC2(n=1,delta=0.4,theta=0.5,model="IEVL")
#'rC2(n=1,delta=0.4,theta=c(1,4),model="IEVct")
#' @export

rC2<-function(n,delta,theta,model,scale="unif")
{
  d<-delta
  R<-1/runif(n)
  if(model=="IEVL")
  {
    require(evd)
    Z<-1/evd::rbvevd(n,model="log",dep=theta,mar1=c(1,1,1))
    W<-exp(Z)
  }
  else if(model=="IEVct")
  {
    require(evd)
    Z<-1/evd::rbvevd(n,model="ct",alpha=theta[1],beta=theta[2],mar1=c(1,1,1))
    W<-exp(Z)
  }
  else if(model=="Gauss")
  {
    require(mvtnorm)
    Z<-mvtnorm::rmvnorm(n,sigma=matrix(c(1,theta,theta,1),2,2))
    W<-1/(1-pnorm(Z))
  }
  Y<-d*log(R) +(1-d)*log(W)
  if(scale=="unif"){return(t(apply(Y,1,F.m.y,d=d)))}
  else if(scale=="y"){return(Y)}
  else if(scale=="x"){return(exp(Y))}
}


##############################################################################
#' Function to fit the 2-dimensional copula with censored likelihood

#' @param datU n * 2 matrix of data on uniform margins
#' @param thresh  bivariate censoring threshold in (0,1)^2
#' @param init.val parameter vector of (delta, theta), where theta is the parameter(s) for the W model
#' @param cens.type either "partial" (whereby only components exceeding the threshold in all margins use the full density contribution) or "max"
#' (whereby components that exceed the threshold in at least one margin use the full density contribution)
#' @param model  character "IEVL" for inverted extreme value logistic; "IEVct" for inverted extreme value Coles_tawn dirichlet, "Gauss" for BV normal
#' @param optim logical. If TRUE then function optimizes; if FALSE then the negative log likelihood is returned.
#' @param std.err  logical; calculate hessian and return standard errors?
#' @param ncore  number of cores to use. If >1 then requires multicore package
#' @param reltol optimization tolerance. Larger values produce less accurate, but faster, optimization.
#' @param ... additional arguments to be passed to optim
#' @return if optim=TRUE returns list with objects "mle", "nll" and "convergence"; optionally also "var.cov" and "se". If optim=FALSE returns the negative log-likelihood
#' @author Jenny Wadsworth
#' @export

fit.cop.2dim<-function(datU, thresh, init.val, cens.type=c("max","partial"), model, optim=TRUE, std.err=FALSE, ncore=1, reltol=1e-4, ...)
{
  require(evd)
  require(mvtnorm)

  if(optim){
  if(ncore>1)
  {
    require(multicore)
    if(cens.type=="max"){opt<-optim(nll.cens.y.mc,u=datU,par=init.val,thresh=thresh,control=list(reltol=reltol),model=model,ncore=ncore,hessian=std.err,...)}
    else if(cens.type=="partial"){opt<-optim(nll.cens.y2.mc,u=datU,par=init.val,thresh=thresh,control=list(reltol=reltol),model=model,ncore=ncore,hessian=std.err,...)}
  }
  else{
    if(cens.type=="max"){opt<-optim(nll.cens.y,u=datU,par=init.val,thresh=thresh,control=list(reltol=reltol),model=model,hessian=std.err,...)}
    else if(cens.type=="partial"){opt<-optim(nll.cens.y2,u=datU,par=init.val,thresh=thresh,control=list(reltol=reltol),model=model,hessian=std.err,...)}
  }
  out<-list()
  out$mle<-opt$par
  out$nll<-opt$value
  out$convergence<-opt$conv
  if(std.err)
  {
    out$var.cov<-solve(opt$hessian)
    out$se<-sqrt(diag(out$var.cov))
  }
  return(out)
  }
  else{
    if(ncore>1)
    {
      require(multicore)
      if(cens.type=="max"){nll<-nll.cens.y.mc(u=datU,par=init.val,thresh=thresh,model=model,ncore=ncore)}
      else if(cens.type=="partial"){nll<-nll.cens.y2.mc(u=datU,par=init.val,thresh=thresh,model=model,ncore=ncore)}
    } else{
      if(cens.type=="max"){nll<-nll.cens.y(u=datU,par=init.val,thresh=thresh,model=model)}
      else if(cens.type=="partial"){nll<-nll.cens.y2(u=datU,par=init.val,thresh=thresh,model=model)}
    }
    return(nll)
  }
}


##############################################################################
# chiu
# calculate \chi(u)

# Arguments:
# u - scalar value of u \in (0,1)
# delta - value of delta
# theta - value of parameter of W model
# model - character "IEVL" for inverted extreme value logistic; "Gauss" for BV normal

chiu<-function(u, delta, theta, model)
{
  y<-NULL
  y[1]<-y[2]<-QF.y(u,d=delta)
  return(Fbar.jt.y(y=y,d=delta,theta=theta,model=model)/(1-u))
}


##############################################################################

#' calculates model-based chi_u in 2 dimensions, assuming W to have either Gaussian or inverted extreme value copulas, as specified
#'
#' @param  u sequence of values at which to calculate chi
#' @param delta value of delta
#' @param theta - value of parameter of W model
#' @param model - character "IEVL" for inverted extreme value logistic; "Gauss" for BV normal
#'
#' @return returns a vector corresponding to the value of chiu at each value of u
#' @author Jenny Wadsworth
#'
#' @export
#'

chiu2<-Vectorize(chiu,vectorize.args = "u")
