#####################################################################
#####################################################################
###                                                               ###
### Censored and uncensored likelihood functions for the RW model ###
###                                                               ###
#####################################################################
#####################################################################

# Renamed function: previously called fitcopulaRWcens

#' Function to fit the D-dimensional copula with censored full likelihood

#' @param datU  n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
#' @param coord D*2 matrix of spatial coordinates
#' @param thresh quantile at which to censor, for example 0.95
#' @param model character vector specifying model for the correlation matrix of the Gaussian process W. AT THIS POINT ONLY THE POWERED EXPONENTIAL MODEL IS IMPLEMENTED
#' @param init.val gives the initial values for all parameters: first the parameter delta of R, and second the parameters of the model for the correlation function
#' @param fixed (must be same length as init) sets some parameters fixed to their initial values. This is used to optimize the likelihood function only with respect to some of the parameters.
#' @param std.err logical. If TRUE, optim will return the hessian
#' @param optim logical. If optim=FALSE, the function returns nllik(init.val) (no optimization). Can be useful to evaluate the calculation time for likelihood evaluation
#' @param method character vector specifying optimization method for optim
#' @param sum logical - if TRUE returns negative log-likelihood for sample; if FALSE then the value of the likelihood for each vector is returned. Not intended to be called by the user.
#' @param print locgical - if TRUE, the parameters and negative log likelihood value will print at each iteration
#' @param ... additional entries for the function optim
#' @return list with objects "mle", "nllik", "convergence", "hessian" and "counts"
#'
#' @author Raphael Huser
#' @export


fit.cop.Ddim <- function(datU,coord,thresh,model='pexp',init.val,fixed=rep(FALSE,3),std.err=FALSE,optim=TRUE,
                         method="Nelder-Mead",sum=TRUE,print=FALSE,rel.tol=NULL,...){

  # to allow easier writing of a function for the sandwich covariance
  #  if(is.vector(datU)){
  #    datU<-matrix(datU,ncol=length(datU))
  #  }
  thres<-thresh
  ### censor data below the threshold
  datUc <- datU; datUc[datU<thres] <- thres

  ### separate the obs depending on the number of exceedances
  n.obs <- nrow(datU)
  n.stat <- ncol(datU)
  Iexc <- apply(datUc>thres,1,which)
  I1 <- which(lapply(Iexc,length)==n.stat) #no censoring
  I2 <- which(lapply(Iexc,length)>0 & lapply(Iexc,length)<n.stat) #some censoring
  nfc <- n.obs-length(I1)-length(I2) #number of fully censored obs

  #### model for W
  D <- as.matrix(dist(coord))
  # correlation functions
  if (model=='pexp'){fcor <- function(h,b){return(exp(-(h/exp(b[1]))^b[2]))}}
  #conditions on the parameters of the correlation function (I need this such that the nllik function returns Inf if the conditions are not met)
  if (model=='pexp'){cond.cor <- function(b){return(b[2]>0 & b[2]<2)}}

  ### negative censored log-likelihood
  nllik <- function(a,b){ #a is delta (weight parameter), b are parameters of W (correlation function)
    #check conditions on parameters
    if (a[1]<=0 | a[1]>=1){return(Inf)}
    if (!cond.cor(b)){return(Inf)}

    #calculate the matrix of correlation in W
    sigmab <- fcor(D,b)

    tr <- try(chol(sigmab),TRUE)
    if (is(tr,"try-error")){return(Inf)}

    #fix random seed (and save the current random seed to restore it at the end)
    if (!exists(".Random.seed", mode="numeric", envir=globalenv()))
      sample(NA)
    oldSeed <- get(".Random.seed", mode="numeric", envir=globalenv())
    set.seed(123456)

    #transform data datUc to RW scale using qG
    xc <- matrix(nrow=n.obs,ncol=n.stat)
    if(any(datUc>thres)){xc[datUc>thres] <- qG1(datUc[datUc>thres],delta=a)}
    xthres <- qG1(thres,delta=a)
    xc[datUc==thres] <- xthres

    #sum the contributions for the different cases
    contrib1 <- contrib2 <- contrib3 <- 0
    if (length(I1)>0){ #no censoring
      if(!sum){contrib1 <- dC(xc[I1,],delta=a,sigma=sigmab,log=TRUE,RWscale=TRUE)
      }else{contrib1 <- sum(dC(xc[I1,],delta=a,sigma=sigmab,log=TRUE,RWscale=TRUE))}
    }
    if (length(I2)>0){ #partial censoring
      if(!sum){contrib2 <- dCI(as.list(data.frame(t(xc)))[I2],delta=a,sigma=sigmab,I=Iexc[I2],log=TRUE,RWscale=TRUE,rel.tol=rel.tol)
      }else{
        contrib2 <- sum(dCI(as.list(data.frame(t(xc)))[I2],delta=a,sigma=sigmab,I=Iexc[I2],log=TRUE,RWscale=TRUE,rel.tol=rel.tol))
      }
    }
    #fully censored obs
    contrib3 <- pC(rep(xthres,n.stat),delta=a,sigma=sigmab,log=TRUE,RWscale=TRUE,rel.tol=rel.tol)

    if(!sum){
      nllvec<-rep(-contrib3,n.obs)
      nllvec[I1]<--contrib1
      nllvec[I2]<--contrib2
    }
    #restore random seed to its previous value
    assign(".Random.seed", oldSeed, envir=globalenv())

    if(sum){
      if(print){
       print(c(a,b,-(contrib1+contrib2+nfc*contrib3)))
      }
      return(-(contrib1+contrib2+nfc*contrib3))
      }
    else{return(nllvec)}
  }

  ### optimization with respect to parameters that are not fixed
  if (optim==TRUE){
    init.val2 <- init.val[which(!fixed)]
    nllik2 <- function(x){
      xx <- init.val
      xx[which(!fixed)] <- x
      return(nllik(a=xx[1],b=xx[-c(1)]))
    }
    opt <- optim(init.val2,nllik2,hessian=std.err,method=method,control=list(...))

    #results
    mle <- c()
    mle[!fixed] <- opt$par
    mle[fixed] <- init.val[fixed]
    z <- list()
    z$mle <- mle
    z$nllik <- opt$val
    z$convergence <- opt$convergence
    z$hessian <- opt$hessian
    z$counts <- opt$counts

    return(z)
  }

  if (optim==FALSE){
    return(nllik(a=init.val[1],b=init.val[-c(1)]))
  }
}




#' Function to fit the D-dimensional copula with censored full likelihood. Uses faster QMC computation than fit.cop.Ddim.

#' @param datU  n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
#' @param coord D*2 matrix of spatial coordinates
#' @param thresh quantile at which to censor, for example 0.95
#' @param model character vector specifying model for the correlation matrix of the Gaussian process W. AT THIS POINT ONLY THE POWERED EXPONENTIAL MODEL IS IMPLEMENTED
#' @param init.val gives the initial values for all parameters: first the parameter delta of R, and second the parameters of the model for the correlation function
#' @param fixed (must be same length as init) sets some parameters fixed to their initial values. This is used to optimize the likelihood function only with respect to some of the parameters.
#' @param std.err logical. If TRUE, optim will return the hessian
#' @param optim logical. If optim=FALSE, the function returns nllik(init.val) (no optimization). Can be useful to evaluate the calculation time for likelihood evaluation
#' @param method character vector specifying optimization method for optim
#' @param sum logical - if TRUE returns negative log-likelihood for sample; if FALSE then the value of the likelihood for each vector is returned. Not intended to be called by the user.
#' @param print logical - if TRUE, the parameters and negative log likelihood value will print at each iteration
#' @param prime a prime number, for calculation of multivariate normal probabilities via QMC methods
#' @param ncore number of cores to use for parallel computation
#' @param ... additional entries for the function optim
#' @return list with objects "mle", "nllik", "convergence", "hessian" and "counts"
#'
#' @references de Fondeville and Davison (2016) High-dimensional peaks-over-threshold inference for the Brown-Resnick process. https://arxiv.org/abs/1605.08558
#' @author Raphael Huser and Jenny Wadsworth
#' @export


fit.cop.Ddim.QMC <- function(datU,coord,thresh,model='pexp',init.val,fixed=rep(FALSE,3),std.err=FALSE,optim=TRUE,
                             method="Nelder-Mead",sum=TRUE,print=FALSE,prime=127,ncore=1,rel.tol=NULL,...){

  require(mvPot)
  if(ncore>1){require(parallel)}
  # to allow easier writing of a function for the sandwich covariance
  #  if(is.vector(datU)){
  #    datU<-matrix(datU,ncol=length(datU))
  #  }
  thres<-thresh
  ### censor data below the threshold
  datUc <- datU; datUc[datU<thres] <- thres

  ### separate the obs depending on the number of exceedances
  n.obs <- nrow(datU)
  n.stat <- ncol(datU)
  Iexc <- apply(datUc>thres,1,which)
  I1 <- which(lapply(Iexc,length)==n.stat) #no censoring
  I2 <- which(lapply(Iexc,length)>0 & lapply(Iexc,length)<n.stat) #some censoring
  nfc <- n.obs-length(I1)-length(I2) #number of fully censored obs

  #### model for W
  D <- as.matrix(dist(coord))
  # correlation functions
  if (model=='pexp'){fcor <- function(h,b){return(exp(-(h/exp(b[1]))^b[2]))}}
  #conditions on the parameters of the correlation function (I need this such that the nllik function returns Inf if the conditions are not met)
  if (model=='pexp'){cond.cor <- function(b){return(b[2]>0 & b[2]<2)}}

  ### negative censored log-likelihood
  nllik <- function(a,b){ #a is delta (weight parameter), b are parameters of W (correlation function)
    #check conditions on parameters
    if (a[1]<=0 | a[1]>=1){return(Inf)}
    if (!cond.cor(b)){return(Inf)}

    #calculate the matrix of correlation in W
    sigmab <- fcor(D,b)

    tr <- try(chol(sigmab),TRUE)
    if (is(tr,"try-error")){return(Inf)}

    #fix random seed (and save the current random seed to restore it at the end)
    if (!exists(".Random.seed", mode="numeric", envir=globalenv()))
      sample(NA)
    oldSeed <- get(".Random.seed", mode="numeric", envir=globalenv())
    set.seed(123456)

    #transform data datUc to RW scale using qG
    xc <- matrix(nrow=n.obs,ncol=n.stat)
    if(any(datUc>thres)){xc[datUc>thres] <- qG1(datUc[datUc>thres],delta=a)}
    xthres <- qG1(thres,delta=a)
    xc[datUc==thres] <- xthres

    #sum the contributions for the different cases
    contrib1 <- contrib2 <- contrib3 <- 0
    if (length(I1)>0){ #no censoring
      if(!sum){contrib1 <- dC(xc[I1,],delta=a,sigma=sigmab,log=TRUE,RWscale=TRUE)
      }else{contrib1 <- sum(dC(xc[I1,],delta=a,sigma=sigmab,log=TRUE,RWscale=TRUE))}
    }
    if (length(I2)>0){ #partial censoring
      if(!sum){contrib2 <- dCI.QMC(as.list(data.frame(t(xc)))[I2],delta=a,sigma=sigmab,I=Iexc[I2],log=TRUE,RWscale=TRUE,prime=prime,ncore=ncore,rel.tol=rel.tol)
      }else{
        contrib2 <- sum(dCI.QMC(as.list(data.frame(t(xc)))[I2],delta=a,sigma=sigmab,I=Iexc[I2],log=TRUE,RWscale=TRUE,prime=prime,ncore=ncore,rel.tol=rel.tol))
      }
    }
    #fully censored obs
    contrib3 <- pC.QMC(rep(xthres,n.stat),delta=a,sigma=sigmab,log=TRUE,RWscale=TRUE,prime=prime,rel.tol=rel.tol)

    if(!sum){
      nllvec<-rep(-contrib3,n.obs)
      nllvec[I1]<--contrib1
      nllvec[I2]<--contrib2
    }
    #restore random seed to its previous value
    assign(".Random.seed", oldSeed, envir=globalenv())

    if(sum){
      if(print){
        print(c(a,b,-(contrib1+contrib2+nfc*contrib3)))
      }
      return(-(contrib1+contrib2+nfc*contrib3))
    }
    else{return(nllvec)}
  }

  ### optimization with respect to parameters that are not fixed
  if (optim==TRUE){
    init.val2 <- init.val[which(!fixed)]
    nllik2 <- function(x){
      xx <- init.val
      xx[which(!fixed)] <- x
      return(nllik(a=xx[1],b=xx[-c(1)]))
    }
    opt <- optim(init.val2,nllik2,hessian=std.err,method=method,control=list(...))

    #results
    mle <- c()
    mle[!fixed] <- opt$par
    mle[fixed] <- init.val[fixed]
    z <- list()
    z$mle <- mle
    z$nllik <- opt$val
    z$convergence <- opt$convergence
    z$hessian <- opt$hessian
    z$counts <- opt$counts

    return(z)
  }

  if (optim==FALSE){
    return(nllik(a=init.val[1],b=init.val[-c(1)]))
  }
}



##### The function to fit the copula with CENSORED PAIRWISE LIKELIHOOD
fitcopulaRWcenspair <- function(datU,coord,maxD,thres,model='pexp',init.val,fixed=rep(FALSE,3),hessian=FALSE,optim=TRUE,method="Nelder-Mead",...){
	#datU is the n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
	#coord is D*2
	#thres is a probability, for example 0.95
	#model is the model for the correlation matrix of the Gaussian process W. AT THIS POINT ONLY THE POWERED EXPONENTIAL MODEL IS IMPLEMENTED
	#init.val gives the initial values for all parameters: first the parameter delta of R, and second the parameters of the model for the correlation function
	#fixed (must be same length as init) sets some parameters fixed to their initial values. This is used to optimize the likelihood function only with respect to some of the parameters.
	#hessian is for the function optim
	#if optim=FALSE, the function returns nllik(init.val) (no optimization). Can be useful to evaluate the calculation time for likelihood evaluation
	#... are additional entries for the function optim

	### distance matrix
	D <- as.matrix(dist(coord))

	### censor data below the threshold
	datUc <- datU; datUc[datU<thres] <- thres

	### separate the obs depending on the number of exceedances
	n.obs <- nrow(datU)
	n.stat <- ncol(datU)
	#all.pairs <- combn(n.stat,2,simplify=FALSE)
	all.pairs <- lapply(apply(cbind(row(D)[which(D<=maxD & row(D)<col(D))],col(D)[which(D<=maxD & row(D)<col(D))]),1,list),unlist)

	datU.pairs <- datUc.pairs <- Iexc <- I1 <- I2 <- nfc <- list()
	for(i in 1:length(all.pairs)){
		datU.pairs[[i]] <- datU[,all.pairs[[i]]]
		datUc.pairs[[i]] <- datUc[,all.pairs[[i]]]
		Iexc[[i]] <- apply(datUc.pairs[[i]]>thres,1,which)
		I1[[i]] <- which(lapply(Iexc[[i]],length)==2) #no censoring
		I2[[i]] <- which(lapply(Iexc[[i]],length)==1) #partial censoring
		nfc[[i]] <- n.obs-length(I1[[i]])-length(I2[[i]]) #number of fully censored obs
	}

	#### model for W
	# correlation functions
	if (model=='pexp'){fcor <- function(h,b){return(exp(-(h/exp(b[1]))^b[2]))}}
	#conditions on the parameters of the correlation function (I need this such that the nllik function returns Inf if the conditions are not met)
	if (model=='pexp'){cond.cor <- function(b){return(b[2]>0 & b[2]<2)}}

	### negative censored log-likelihood
	npllik <- function(a,b){ #a is delta (weight parameter), b are parameters of W (correlation function)
		#check conditions on parameters
		if (a[1]<=0 | a[1]>=1){return(Inf)}
		if (!cond.cor(b)){return(Inf)}

		#calculate the matrix of correlation in W
		sigmab <- fcor(D,b)
		sigmab.pairs <- list()
		for(i in 1:length(all.pairs)){
			sigmab.pairs[[i]] <- sigmab[all.pairs[[i]],all.pairs[[i]]]
			tr <- try(chol(sigmab.pairs[[i]]),TRUE)
			if (is(tr,"try-error")){return(Inf)}
		}


		#fix random seed (and save the current random seed to restore it at the end)
		oldSeed <- get(".Random.seed", mode="numeric", envir=globalenv())
		set.seed(123456)

		#transform data datUc to RW scale using qG
		xc <- matrix(nrow=n.obs,ncol=n.stat)
		xc[datUc>thres] <- qG1(datUc[datUc>thres],delta=a)
		xthres <- qG1(thres,delta=a)
		xc[datUc==thres] <- xthres

		xc.pairs <- list()
		for(i in 1:length(all.pairs)){
			xc.pairs[[i]] <- xc[,all.pairs[[i]]]
		}

		#sum the pairwise contributions for the different cases
		contrib1 <- contrib2 <- contrib3 <- 0
		for(i in 1:length(all.pairs)){
			if (length(I1[[i]])>0){ #no censoring
				contrib1 <- contrib1 + sum(dC(xc.pairs[[i]][I1[[i]],],delta=a,sigma=sigmab.pairs[[i]],log=TRUE,RWscale=TRUE))
			}
			if (length(I2[[i]])>0){ #partial censoring
				contrib2 <- contrib2 + sum(dCI(as.list(data.frame(t(xc.pairs[[i]])))[I2[[i]]],delta=a,sigma=sigmab.pairs[[i]],I=Iexc[[i]][I2[[i]]],log=TRUE,RWscale=TRUE))
			}
			#fully censored obs
			contrib3 <- contrib3 + nfc[[i]]*pC(rep(xthres,2),delta=a,sigma=sigmab.pairs[[i]],log=TRUE,RWscale=TRUE)
		}

		#restore random seed to its previous value
		assign(".Random.seed", oldSeed, envir=globalenv())

		return(-(contrib1+contrib2+contrib3))
	}

	### optimization with respect to parameters that are not fixed
	if (optim==TRUE){
		init.val2 <- init.val[which(!fixed)]
		npllik2 <- function(x){
			xx <- init.val
			xx[which(!fixed)] <- x
			return(npllik(a=xx[1],b=xx[-c(1)]))
		}
		opt <- optim(init.val2,npllik2,hessian=hessian,method=method,control=list(...))

		#results
		mle <- c()
		mle[!fixed] <- opt$par
		mle[fixed] <- init.val[fixed]
		z <- list()
		z$mle <- mle
		z$nllik <- opt$val
		z$convergence <- opt$convergence
		z$hessian <- opt$hessian
		z$counts <- opt$counts

		return(z)
	}

	if (optim==FALSE){
		return(npllik(a=init.val[1],b=init.val[-c(1)]))
	}
}




##### The function to fit the copula with NON-CENSORED FULL LIKELIHOOD
fitcopulaRWnocens <- function(datU,coord,model='pexp',init.val,fixed=rep(FALSE,3),hessian=FALSE,optim=TRUE,method="Nelder-Mead",...){
	#datU is the n*D matrix of data (on the uniform scale): n obs of a D dimensional vector
	#coord is D*2
	#model is the model for the correlation matrix of the Gaussian process W. AT THIS POINT ONLY THE POWERED EXPONENTIAL MODEL IS IMPLEMENTED
	#init.val gives the initial values for all parameters: first the parameter delta of R, and second the parameters of the model for the correlation function
	#fixed (must be same length as init) sets some parameters fixed to their initial values. This is used to optimize the likelihood function only with respect to some of the parameters.
	#hessian is for the function optim
	#if optim=FALSE, the function returns nllik(init.val) (no optimization). Can be useful to evaluate the calculation time for likelihood evaluation
	#... are additional entries for the function optim

	#### model for W
	D <- as.matrix(dist(coord))
	# correlation functions
	if (model=='pexp'){fcor <- function(h,b){return(exp(-(h/exp(b[1]))^b[2]))}}
	#conditions on the parameters of the correlation function (I need this such that the nllik function returns Inf if the conditions are not met)
	if (model=='pexp'){cond.cor <- function(b){return(b[2]>0 & b[2]<2)}}

	### negative censored log-likelihood
	nllik <- function(a,b){ #a is delta (weight parameter), b are parameters of W (correlation function)
		#check conditions on parameters
		if (a[1]<=0 | a[1]>=1){return(Inf)}
		if (!cond.cor(b)){return(Inf)}

		#calculate the matrix of correlation in W
		sigmab <- fcor(D,b)

		tr <- try(chol(sigmab),TRUE)
                if (is(tr,"try-error")){return(Inf)}

		#fix random seed (and save the current random seed to restore it at the end)
		oldSeed <- get(".Random.seed", mode="numeric", envir=globalenv())
		set.seed(123456)

		#transform data datUc to RW scale using qG
		x <- matrix(qG1(c(datU),delta=a),ncol=ncol(datU))

		contrib <- sum(dC(x,delta=a,sigma=sigmab,log=TRUE,RWscale=TRUE))

		#restore random seed to its previous value
		assign(".Random.seed", oldSeed, envir=globalenv())

		return(-contrib)
	}

	### optimization with respect to parameters that are not fixed
	if (optim==TRUE){
		init.val2 <- init.val[which(!fixed)]
		nllik2 <- function(x){
			xx <- init.val
			xx[which(!fixed)] <- x
			return(nllik(a=xx[1],b=xx[-c(1)]))
		}
		opt <- optim(init.val2,nllik2,hessian=hessian,method=method,control=list(...))

		#results
		mle <- c()
		mle[!fixed] <- opt$par
	    	mle[fixed] <- init.val[fixed]
		z <- list()
		z$mle <- mle
		z$nllik <- opt$val
		z$convergence <- opt$convergence
		z$hessian <- opt$hessian
		z$counts <- opt$counts

		return(z)
	}

	if (optim==FALSE){
		return(nllik(a=init.val[1],b=init.val[-c(1)]))
	}
}


