#############################################################################################################################################
### Marginal Distribution, Quantile, Density Functions, and Random generator for the Gaussian scale mixture model X=R^(delta)*W^(1-delta) ###
#############################################################################################################################################

#Marginal distribution function (CDF) !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
pG1 <- function(x,delta,log=FALSE,lower.tail=TRUE){
	if(delta!=1/2){
		if(lower.tail){
			logres <- log( 1-(delta/(2*delta-1))*exp(-x/delta) + ((1-delta)/(2*delta-1))*exp(-x/(1-delta)) )
		} else{
			logres <- log( (delta/(2*delta-1))*exp(-x/delta) - ((1-delta)/(2*delta-1))*exp(-x/(1-delta)) )
		}
	} else{
		if(lower.tail){
			logres <- log( 1-exp(-2*x)*(1+2*x) )
		} else{
			logres <- -2*x+log(1+2*x)
		}
	}
	if(log){
		return(logres)
	} else{
		return(exp(logres))
	}
}

#Marginal density function (PDF) !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
dG1 <- function(x,delta,log=FALSE){
	if(delta!=1/2){
		logres <- log( (1/(2*delta-1))*(exp(-x/delta) - exp(-x/(1-delta))) )
	} else{
		logres <- log(4)+log(x)-2*x
	}
	if(log){
		return(logres)
	} else{
		return(exp(logres))
	}
}

#Marginal quantile function !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
qG1 <- function(p,delta,log=FALSE){
	pmat <- p
	if(!is.matrix(p)){
		pmat <- matrix(p,nrow=1)
	}
	n <- nrow(pmat)
	D <- ncol(pmat)
	fun <- function(x,p,delta){
		return( pG1(exp(x),delta,log=TRUE,lower.tail=FALSE)-log(1-p) )
	}
	logres <- matrix(nrow=n,ncol=D)
	for (i in 1:n){
		for(j in 1:D){
			pi <- pmat[i,j]
			if(pi<=0){
				logres[i,j] <- -Inf
			} else if(pi>=1){
				logres[i,j] <- Inf
			} else{
				logres[i,j] <- uniroot(fun,interval=c(-5,5),p=pi,delta=delta,extendInt='yes')$root
			}
		}
	}
	if(!is.matrix(p)){
		logres <- as.vector(logres)
	}
	if(log){
		return(logres)
	} else{
		return(exp(logres))
	}
}

#Random generator from marginal distribution G1 !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
rG1 <- function(n,delta){
	logR <- rexp(n)
	logW <- rexp(n)
	X <- delta*logR + (1-delta)*logW
	return(X)
}



###################################################################################################################################################################################
### Multivariate Distribution function, Partial derivatives, Density Function, and Random generator for the Gaussian scale mixture model Y=log(X)=delta*log(R)+(1-delta)*log(W) ###
###################################################################################################################################################################################

#Multivariate distribution function (CDF) !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
pG <- function(x,delta,sigma,log=FALSE,rel.tol){ ### x is an nxD matrix; if x is a vector, it is interpreted as a single D-variate vector (not D independent univariate random variables)
	if(!is.matrix(x)){
		x <- matrix(x,nrow=1)
	}
	D <- ncol(x)
	pGi <- function(xi){
	  fun <- function(p,delta){
	    Y1 <- xi/(1-delta)+delta*log(1-p)/(1-delta)
	    Y2 <- Y1
	    Y1small <- Y1<qexp(0.5)
	    Y2[Y1small] <- qnorm(1-exp(-Y1[Y1small]))
	    Y2[!Y1small] <- -qnorm(exp(-Y1[!Y1small]))
	    set.seed(123) # to give "same realization" of pmvnorm each time - helps integral converge (much) more rapidly
	    return(pmvnorm(upper=Y2,sigma=sigma)[1])
	  }
	  fun<-Vectorize(fun,vectorize.args = "p")

	  if(is.null(rel.tol)){
		if(delta <= 0.55){
                        rel.tol <- 10^(-4)
                } else if(delta > 0.55 & delta <= 0.75){
                        rel.tol <- 10^(-5)
                } else if(delta > 0.75 & delta <= 0.85){
                        rel.tol <- 10^(-7)
                } else if(delta > 0.85){
			rel.tol <- 10^(-8)
		}}
		res <- integrate(fun,lower=0,upper=pexp(min(xi)/delta),delta,rel.tol=rel.tol,stop.on.error=FALSE)$value
		return(res)
	}
	logres <- log(apply(x,1,pGi))
	if(log){
		return( logres )
	} else{
		return( exp(logres) )
	}
}


pG.QMC <- function(x,delta,sigma,log=FALSE,prime,rel.tol){ ### x is an nxD matrix; if x is a vector, it is interpreted as a single D-variate vector (not D independent univariate random variables)
  if(!is.matrix(x)){
    x <- matrix(x,nrow=1)
  }
  D <- ncol(x)
  pGi <- function(xi){
    seed<-sample(1:10000,size=1)
    fun <- function(p,delta){
      Y1 <- xi/(1-delta)+delta*log(1-p)/(1-delta)
      Y2 <- Y1
      Y1small <- Y1<qexp(0.5)
      Y2[Y1small] <- qnorm(1-exp(-Y1[Y1small]))
      Y2[!Y1small] <- -qnorm(exp(-Y1[!Y1small]))
      vec <- genVecQMC(prime, D)
      set.seed(seed = seed)
      return(mvPot::mvtNormQuasiMonteCarlo(upper=Y2,cov=sigma,p=vec$primeP,genVec=vec$genVec)[1])
    }
    fun<-Vectorize(fun,vectorize.args = "p")

    if(is.null(rel.tol)){
    if(delta <= 0.55){
      rel.tol <- 10^(-4)
    } else if(delta > 0.55 & delta <= 0.75){
      rel.tol <- 10^(-5)
    } else if(delta > 0.75 & delta <= 0.85){
      rel.tol <- 10^(-7)
    } else if(delta > 0.85){
      rel.tol <- 10^(-8)
    }}
    res <- integrate(fun,lower=0,upper=pexp(min(xi)/delta),delta,rel.tol=rel.tol,stop.on.error=FALSE)$value
    return(res)
  }
  logres <- log(apply(x,1,pGi))
  if(log){
    return( logres )
  } else{
    return( exp(logres) )
  }
}

#Multivariate density function (PDF) !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
dG <- function(x,delta,sigma,log=FALSE){
	if(!is.matrix(x)){
		x <- matrix(x,nrow=1)
	}
	D <- ncol(x)
	dGi <- function(xi){
	  fun <- function(p,delta){
	    X <- matrix(xi,ncol=D,nrow=length(p),byrow=TRUE)
	    Y1 <- X/(1-delta)+delta*log(1-p)/(1-delta);
	    ind.na <- apply(Y1<=0,1,any)
	    Y1[ind.na,] <- NA
	    #Y2 <- qnorm(pexp(Y1))
	    Y2 <- Y1
	    Y1small <- Y1<qexp(0.5)
	    Y2[Y1small] <- qnorm(1-exp(-Y1[Y1small]))
	    Y2[!Y1small] <- -qnorm(exp(-Y1[!Y1small]))
	    Y3 <- -log(1-delta)+dexp(Y1,log=TRUE)-dnorm(Y2,log=TRUE)
	    Y4 <- c()
	    Y4[!ind.na] <- exp(apply(matrix(Y2[!ind.na,],ncol=D),1,function(x){return(dmvnorm(x=x,sigma=sigma,log=TRUE)[1])})+rowSums(matrix(Y3[!ind.na,],ncol=D)))
	    Y4[ind.na] <- 0
	    return( Y4 )
	  }
		if(delta <= 0.55){
                        rel.tol <- 10^(-4)
                } else if(delta > 0.55 & delta <= 0.75){
                        rel.tol <- 10^(-5)
                } else if(delta > 0.75 & delta <= 0.85){
                        rel.tol <- 10^(-7)
                } else if(delta > 0.85){
                        rel.tol <- 10^(-8)
                }
		res <- integrate(fun,lower=0,upper=pexp(min(xi)/delta),delta,rel.tol=rel.tol,stop.on.error=FALSE)$value
		return(res)
	}
	logres <- log(apply(x,1,dGi))
	if(log){
		return( logres )
	} else{
		return( exp(logres) )
	}
}

#Partial derivatives of the distribution function !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
dGI <- function(x,delta,sigma,I,log=FALSE,rel.tol){
	if(!is.list(x)){
		if(!is.matrix(x)){
			x <- matrix(x,nrow=1)
		}
		x <- as.list(data.frame(t(x)))
	}
	D <- length(x[[1]])
	n <- length(x)
	#I is the index vector for partial derivatives
	if(!is.list(I)){
		I <- matrix(I,n,length(I),byrow=TRUE)
		I <- as.list(data.frame(t(I)))
	}

	dGIi <- function(xi,I){
		nI <- length(I)
		#parameters for the conditional distribution of x[I^c] | x[I]
		sigma_II <- matrix(sigma[I,I],ncol=nI) #need to force a matrix in case nI=1
		sigma_II_inv <- solve(sigma_II)
		sigma_IcI <- matrix(sigma[-I,I],ncol=nI) #same problem here
		sigma_IcIc <- matrix(sigma[-I,-I],ncol=D-nI)
		sigma_IIc <- t(sigma_IcI)
		sig1 <- sigma_IcIc-sigma_IcI%*%sigma_II_inv%*%sigma_IIc

		sig1<-(sig1+t(sig1))/2 # otherwise can get numerically non-symmetric matrices

		fun <- function(p,delta){
		  Y1 <- xi/(1-delta)+delta*log(1-p)/(1-delta);
		  ind.na <- any(Y1<=0)

		  if(!ind.na){
		    Y2 <- Y1
		    Y1small <- Y1<qexp(0.5)
		    Y2[Y1small] <- qnorm(1-exp(-Y1[Y1small]))
		    Y2[!Y1small] <- -qnorm(exp(-Y1[!Y1small]))

		    mu1 <- t(sigma_IcI%*%(sigma_II_inv%*%(Y2[I])))
		    Y3 <- -log(1-delta)+dexp(matrix(Y1[I],ncol=nI),log=TRUE)-dnorm(matrix(Y2[I],ncol=nI),log=TRUE)
		    Y4 <- c()
		    Y4[!ind.na] <- apply(matrix(Y2[I],ncol=nI),1,function(x){return(dmvnorm(x=x,sigma=sigma_II,log=TRUE)[1])})+rowSums(matrix(Y3,ncol=nI))
		    if(nI<D){
		      set.seed(123)  # to give same realization of pmvnorm each time - helps integral converge much more rapidly
		      Y4 <- Y4+ log(pmax(0,pmin(1,pmvnorm(upper=c(Y2[-I]-mu1),sigma=sig1)[1])))
		    }
		    Y4 <- exp(as.numeric(Y4))
		  }
		  else{Y4 <- 0}
		  return( Y4 )
		}
		fun<-Vectorize(fun,vectorize.args = "p")

		if(is.null(rel.tol)){
		if(delta <= 0.55){
                        rel.tol <- 10^(-4)
                } else if(delta > 0.55 & delta <= 0.75){
                        rel.tol <- 10^(-5)
                } else if(delta > 0.75 & delta <= 0.85){
                        rel.tol <- 10^(-7)
                } else if(delta > 0.85){
                        rel.tol <- 10^(-8)
                }}
		res <- integrate(fun,lower=0,upper=pexp(min(xi)/delta),delta,rel.tol=rel.tol,stop.on.error=FALSE)$value
		return(res)
	}
	logres <- log(mapply(dGIi,xi=x,I=I))

	if(log){
		return( logres )
	} else{
		return( exp(logres) )
	}
}


#Partial derivatives of the distribution function !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
dGI.QMC <- function(x,delta,sigma,I,log=FALSE,prime,ncore,rel.tol){
  if(!is.list(x)){
    if(!is.matrix(x)){
      x <- matrix(x,nrow=1)
    }
    x <- as.list(data.frame(t(x)))
  }
  D <- length(x[[1]])
  n <- length(x)
  #I is the index vector for partial derivatives
  if(!is.list(I)){
    I <- matrix(I,n,length(I),byrow=TRUE)
    I <- as.list(data.frame(t(I)))
  }

  dGIi <- function(xi,I,rseed){
    nI <- length(I)
    #parameters for the conditional distribution of x[I^c] | x[I]
    sigma_II <- matrix(sigma[I,I],ncol=nI) #need to force a matrix in case nI=1
    sigma_II_inv <- solve(sigma_II)
    sigma_IcI <- matrix(sigma[-I,I],ncol=nI) #same problem here
    sigma_IcIc <- matrix(sigma[-I,-I],ncol=D-nI)
    sigma_IIc <- t(sigma_IcI)
    sig1 <- sigma_IcIc-sigma_IcI%*%sigma_II_inv%*%sigma_IIc

    sig1<-(sig1+t(sig1))/2 # otherwise can get numerically non-symmetric matrices
    #seed<-sample(1:10000,size=1)

    fun <- function(p,delta){
      Y1 <- xi/(1-delta)+delta*log(1-p)/(1-delta);
      ind.na <- any(Y1<=0)

      if(!ind.na){
        Y2 <- Y1
        Y1small <- Y1<qexp(0.5)
        Y2[Y1small] <- qnorm(1-exp(-Y1[Y1small]))
        Y2[!Y1small] <- -qnorm(exp(-Y1[!Y1small]))

        mu1 <- t(sigma_IcI%*%(sigma_II_inv%*%(Y2[I])))
        Y3 <- -log(1-delta)+dexp(matrix(Y1[I],ncol=nI),log=TRUE)-dnorm(matrix(Y2[I],ncol=nI),log=TRUE)
        Y4 <- c()
        Y4[!ind.na] <- apply(matrix(Y2[I],ncol=nI),1,function(x){return(dmvnorm(x=x,sigma=sigma_II,log=TRUE)[1])})+rowSums(matrix(Y3,ncol=nI))
        if(nI<D){
          set.seed(seed = rseed)
          if(dim(sig1)[1]>1)
          {
            vec <- genVecQMC(prime, D-nI)
            Y4 <- Y4+ log(pmax(0,pmin(1,mvPot::mvtNormQuasiMonteCarlo(upper=c(Y2[-I]-mu1),cov=sig1,p=vec$primeP,genVec=vec$genVec)[1])))
          } else{
            Y4 <- Y4+ log(pmax(0,pmin(1,pnorm(c(Y2[-I]-mu1),sd=sqrt(c(sig1))))))
          }
        }
        Y4 <- exp(as.numeric(Y4))
      }
      else{Y4 <- 0}
      return( Y4 )
    }
    fun<-Vectorize(fun,vectorize.args = "p")

    if(is.null(rel.tol)){
    if(delta <= 0.55){
      rel.tol <- 10^(-4)
    } else if(delta > 0.55 & delta <= 0.75){
      rel.tol <- 10^(-5)
    } else if(delta > 0.75 & delta <= 0.85){
      rel.tol <- 10^(-7)
    } else if(delta > 0.85){
      rel.tol <- 10^(-8)
    }}
    res <- integrate(fun,lower=0,upper=pexp(min(xi)/delta),delta,rel.tol=rel.tol,stop.on.error=FALSE)$value
    return(res)
  }
  #logres <- log(mapply(dGIi,xi=x,I=I))
  rseed<-as.list(sample(1:100000,size=length(I),replace = FALSE))
  logres <- log(parallel::mcmapply(dGIi,xi=x,I=I,rseed=rseed,mc.cores=ncore,mc.set.seed = FALSE)) ###

  if(log){
    return( logres )
  } else{
    return( exp(logres) )
  }
}

#Random generator from the joint distribution G !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
rG <- function(n,delta,sigma){
	logR <- rexp(n)
	logW <- qexp(pnorm(rmvnorm(n,sigma=sigma)))
	X <- delta*logR + (1-delta)*logW
	return(X)
}


#####################################################################################################################################################################
### Copula, Partial derivatives, Copula Density Function, and Copula random generator for the Gaussian scale mixture model Y=log(X)=delta*log(R)+(1-delta)*log(W) ###
#####################################################################################################################################################################

#Copula distribution (CDF)
pC <- function(u,delta,sigma,log=FALSE,RWscale=FALSE,rel.tol){
	if(RWscale){ ### When u is already on the scale of the process RW
		return( pG(u,delta,sigma,log,rel.tol=rel.tol) )
	} else{
		return( pG(qG1(u,delta),delta,sigma,log,rel.tol=rel.tol) )
	}
}

pC.QMC <- function(u,delta,sigma,log=FALSE,RWscale=FALSE,prime,rel.tol){
  if(RWscale){ ### When u is already on the scale of the process RW
    return( pG.QMC(u,delta,sigma,log,prime=prime,rel.tol=rel.tol) )
  } else{
    return( pG.QMC(qG1(u,delta),delta,sigma,log,prime=prime,rel.tol=rel.tol) )
  }
}


#Copula density (PDF) !!! implemented for Y=log(X)=delta*log(R)+(1-delta)*log(W)
dC <- function(u,delta,sigma,log=FALSE,RWscale=FALSE){
	if(!is.matrix(u)){
		u <- matrix(u,nrow=1)
	}
	if(RWscale){ ### When u is already on the scale of the process RW
		val <- dG(u,delta,sigma,log) - rowSums(dG1(u,delta,log=TRUE))
	} else{
		val <- dG(qG1(u,delta),delta,sigma,log=TRUE) - rowSums(dG1(qG1(u,delta),delta,log=TRUE))
	}
	if(!log){
		val <- exp(val)
	}
	return( val )
}

#Partial derivatives of the copula distribution C
dCI <- function(u,delta,sigma,I,log=FALSE,RWscale=FALSE,rel.tol){
	if(!is.list(u)){
		if(!is.matrix(u)){
			u <- matrix(u,nrow=1)
		}
		u <- as.list(data.frame(t(u)))
	}
	n <- length(u)
	#I is the index vector for partial derivatives
	if(!is.list(I)){
		I <- matrix(I,n,length(I),byrow=TRUE)
		I <- as.list(data.frame(t(I)))
	}
	fun1 <- function(u,I){
		return( sum(dG1(u[I],delta,log=TRUE)) )
	}
	fun2 <- function(u,I){
		return( sum(dG1(qG1(u[I],delta),delta,log=TRUE)) )
	}
	if(RWscale){ ### When u is already on the scale of the process RW
		val <- dGI(u,delta,sigma,I,log=TRUE,rel.tol=rel.tol) - mapply(fun1,u=u,I=I)
	} else{
		val <- dGI(lapply(u,qG1,delta=delta),delta,sigma,I,log=TRUE,rel.tol) - mapply(fun2,u=u,I=I)
	}
	if(!log){
		val <- exp(val)
	}
	return( val )
}


dCI.QMC <- function(u,delta,sigma,I,log=FALSE,RWscale=FALSE,prime,ncore,rel.tol){
  if(!is.list(u)){
    if(!is.matrix(u)){
      u <- matrix(u,nrow=1)
    }
    u <- as.list(data.frame(t(u)))
  }
  n <- length(u)
  #I is the index vector for partial derivatives
  if(!is.list(I)){
    I <- matrix(I,n,length(I),byrow=TRUE)
    I <- as.list(data.frame(t(I)))
  }
  fun1 <- function(u,I){
    return( sum(dG1(u[I],delta,log=TRUE)) )
  }
  fun2 <- function(u,I){
    return( sum(dG1(qG1(u[I],delta),delta,log=TRUE)) )
  }
  if(RWscale){ ### When u is already on the scale of the process RW
    val <- dGI.QMC(u,delta,sigma,I,log=TRUE,prime=prime,ncore=ncore,rel.tol=rel.tol) - mapply(fun1,u=u,I=I)
  } else{
    val <- dGI.QMC(lapply(u,qG1,delta=delta),delta,sigma,I,log=TRUE,prime=prime,ncore=ncore,rel.tol=rel.tol) - mapply(fun2,u=u,I=I)
  }
  if(!log){
    val <- exp(val)
  }
  return( val )
}

#' Simulate realizations with the d-dimensional copula R^(delta)*W^(1-delta) for W a Gaussian random field
#' @param n number of realizations
#' @param delta value of delta
#' @param sigma covariance matrix of W (d*d)
#' @param empirMar logical; if TRUE translation to uniform scale is done using empirical cdf, else the true cdf is used
#' @author Raphael Huser
#' @examples
#' set.seed(123)
#' coord<-cbind(runif(10,0,5),runif(10,0,5))
#' DM<-as.matrix(dist(coord,upper=T,diag=T))
#' rC(n=1,delta=0.4,sigma=exp(-DM))
#' @export

rC <- function(n,delta,sigma,empirMar=FALSE){
	X <- rG(n,delta,sigma)
	if(empirMar){
		U <- apply(X,2,rank)/(n+1)
	} else{
		U <- pG1(X,delta)
	}
	return(U)
}

###############################################################################################################
### Chi and Chibar coefficients for the Gaussian scale mixture model Y=log(X)=delta*log(R)+(1-delta)*log(W) ###
###############################################################################################################

chi.lim <- function(corr,delta){
	if(delta<=1/2){
		return(0)
	} else if(delta>1/2){
		sigma <- matrix(c(1,corr,corr,1),nrow=2,ncol=2)
		n <- 10^6
		W <- 1/(1-pnorm(rmvnorm(n,sigma=sigma)))
		return( mean(apply(W,1,min)^((1-delta)/delta))*(2*delta-1)/delta )
	}
}

chibar.lim <- function(corr,delta){
	if(delta<1/2){
		eta <- (1+corr)/2
		eta.star <- ifelse(eta/(1+eta)<delta,delta/(1-delta),eta)
		return( 2*eta.star-1 )
	} else if(delta>=1/2){
		return(1)
	}
}

chi <- function(u,corr,delta,Coles.etal=FALSE){
	chis <- rep(NA,length(u))
	sigma <- matrix(c(1,corr,corr,1),2,2)
	for(i in 1:length(u)){
		if(u[i]==1){
			try( chis[i] <- chi.lim(corr,delta) )
		} else{
			if(Coles.etal){
				try( chis[i] <- 2-pC(c(u[i],u[i]),delta,sigma,log=TRUE)/log(u[i]) )
			} else{
				try( chis[i] <- (1-2*u[i]+pC(c(u[i],u[i]),delta,sigma))/(1-u[i]) )
			}
		}
	}
	return(chis)
}

chibar <- function(u,corr,delta){
	chibars <- rep(NA,length(u))
	sigma <- matrix(c(1,corr,corr,1),2,2)
	for(i in 1:length(u)){
		if(u[i]==1){
			try( chibars[i] <- chibar.lim(corr,delta) )
		} else{
			try( chibars[i] <- 2*log(1-u[i])/log(1-2*u[i]+pC(c(u[i],u[i]),delta,sigma))-1 )
		}
	}
	return(chibars)
}

chi.empir <- function(u,corr,delta,n,Coles.etal=FALSE){
	sigma <- matrix(c(1,corr,corr,1),2,2)
	U <- rC(n,delta,sigma,TRUE)
	chis <- rep(NA,length(u))
	for(i in 1:length(u)){
		if(Coles.etal){
			try( chis[i] <- 2-log(mean(U[,1]<=u[i] & U[,2]<=u[i]))/log(u[i]) )
		} else{
			try( chis[i] <- (mean(U[,1]>u[i] & U[,2]>u[i]))/(1-u[i]) )
		}
	}
	return(chis)
}

chibar.empir <- function(u,corr,delta,n){
	sigma <- matrix(c(1,corr,corr,1),2,2)
	U <- rC(n,delta,sigma,TRUE)
	chibars <- rep(NA,length(u))
	for(i in 1:length(u)){
		try( chibars[i] <- 2*log(1-u[i])/log(1-2*u[i]+mean(U[,1]<=u[i] & U[,2]<=u[i]))-1 )
	}
	return(chibars)
}

pC.Gauss <- function(u,Sigma){
	return( pmvnorm(upper=qnorm(u),sigma=Sigma)[1] )
}

chi.Gauss <- function(u,corr,Coles.etal=FALSE){
	chis <- rep(NA,length(u))
	Sigma <- matrix(c(1,corr,corr,1),2,2)
	for(i in 1:length(u)){
		if(u[i]==1){
			try( chis[i] <- 0 )
		} else{
			if(Coles.etal){
				try( chis[i] <- 2-log(pC.Gauss(c(u[i],u[i]),Sigma))/log(u[i]) )
			} else{
				try( chis[i] <- (1-2*u[i]+pC.Gauss(c(u[i],u[i]),Sigma))/(1-u[i]) )
			}
		}
	}
	return(chis)
}

chibar.Gauss <- function(u,corr){
	chi.bars <- rep(NA,length(u))
	Sigma <- matrix(c(1,corr,corr,1),2,2)
	for(i in 1:length(u)){
		if(u[i]==1){
			try( chi.bars[i] <- corr )
		} else{
			try( chi.bars[i] <- 2*log(1-u[i])/log(1-2*u[i]+pC.Gauss(c(u[i],u[i]),Sigma))-1 )
		}
	}
	return(chi.bars)
}





















